<?php
/**
 * LDIE import export class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Export_Data' ) ) {
	/**
	 * Export class
	 */
	abstract class Ld_Admin_Export_Data {

		protected const SITE_URL = '{site_url}';

		protected const SERVER_PATH = '{server_path}';

		/**
		 * Undocumented variable
		 *
		 * @var integer
		 */
		protected $limit = -1;


		/**
		 * Undocumented variable
		 *
		 * @var Class
		 */
		protected $exporter = null;

		/**
		 * Undocumented variable
		 *
		 * @var integer
		 */
		protected $max_chunk = null;

		/**
		 * Undocumented variable
		 *
		 * @var string
		 */
		protected $export_type = null;

		/**
		 * Undocumented variable
		 *
		 * @var Class
		 */
		protected $media = null;

		/**
		 * Undocumented variable
		 *
		 * @var Class
		 */
		protected $fs = null;

		/**
		 * Undocumented variable
		 *
		 * @var array
		 */
		protected $attachments_posts = [];

		/**
		 * Construct
		 */
		public function __construct( $data_type, $export_type, $max_chunk = 150 ) {
			$this->max_chunk   = $max_chunk;
			$this->export_type = $export_type;

			if ( 'json' === $data_type ) {
				$this->exporter = new Ld_Admin_Export_JSON_File();
			} else {
				$this->exporter = new Ld_Admin_Export_XML_File();
			}

			$this->fs = new Ld_Admin_File_System( 'export', $data_type );
			$this->fs->set_is_zip( true );
			$this->media = new Ld_Admin_Export_Media( $this->fs );
		}

		abstract protected function export( $data );

		/**
		 * Function: export_ld_data
		 * call back action from ajax request to export the data
		 *
		 * @since 1.0.0
		 * @throws \Exception When the data is invalid.
		 * @return array
		 */
		public function export_data( $post_types, $course_ids, $content_global_setting, $users_data, $opencourse_checkboxes ) {
			if ( 'per-course' === $this->export_type ) {
				if ( empty( $course_ids ) ) {
					throw new \Exception( esc_html__( 'Select the Courses to export', 'ea_import_export' ) );
				}

				$this->export( $course_ids );
			} else {
				$this->export( $post_types );
			}

			$data_global_content_setting = $this->exporter->global_content_setting( $content_global_setting );
			if ( $data_global_content_setting ) {
				$this->fs->create_file( $data_global_content_setting, 'wp_global_settings' );
			}

			if ( 'yes' === $users_data && 'per-course' !== $this->export_type ) {
				$exported_users_data = [];
				$get_users_data      = $this->get_users_data( $opencourse_checkboxes );

				foreach ( $get_users_data as $user ) {
					$exported_users_data[] = $this->exporter->generate_users_data( $user );
				}

				if ( ! empty( $exported_users_data ) ) {
					$this->fs->create_files( $exported_users_data, 'wp_users_data' );
				}
			}

			if ( 'yes' === $users_data && 'per-course' === $this->export_type ) {
				$exported_users_data = [];

				if ( empty( $course_ids ) ) {
					return;
				}

				$get_user_ids = array();

				foreach ( $course_ids as $course_id ) {
					$get_user_ids[] = learndash_get_course_users_access_from_meta( $course_id );
				}

				$users_id        = array();
				foreach ( $get_user_ids as $get_users ) {

					if ( ! empty( $get_users ) ) {
						foreach ( $get_users as $get_user ) {
							$users_id[] = $get_user;
						}
					}
				}
				$unique_users_ids = array_unique( $users_id );

				$all_users = [];

				foreach ( $unique_users_ids as $key => $unique_users_id ) {
					$users = get_user_by( 'id', $unique_users_id );

					$all_users[ $key ] = (array) $users->data;

					$export_usermeta = $this->get_users_meta_data( $users );

					$all_users[ $key ]['meta'] = $export_usermeta;
				}

				$get_users_data = array_chunk( $all_users, $this->max_chunk, true );

				foreach ( $get_users_data as $get_user_data ) {
					$exported_users_data[] = $this->exporter->generate_users_data( $get_user_data );
				}

				if ( ! empty( $exported_users_data ) ) {
					$this->fs->create_files( $exported_users_data, 'wp_users_data' );
				}
			}

			$exported_categories = $this->get_terms();
			$data_to_save        = $this->exporter->generate_categories_data( $exported_categories );
			if ( $data_to_save ) {
				$this->fs->create_file( $data_to_save, 'wp_categories' );
			}

			$this->media->download_files();

			if ( ! empty( $this->attachments_posts ) ) {
				$attachment_post_to_save = $this->exporter->generate_data( $this->attachments_posts );
				if ( $attachment_post_to_save ) {
					$this->fs->create_file( $attachment_post_to_save, 'attachments' );
				}
			}

			return $this->fs->generate_export_files();

		}

		/**
		 * Get_terms
		 * get all terms available from post
		 *
		 * @since 1.0.0
		 * @param Int $post_id Post ID.
		 * @return array|null
		 */
		protected function get_terms( $post_id = null ) {
			$terms   = get_terms();
			$get_tax = [];

			if ( ! $post_id ) {
				// get all taxonomies.
				foreach ( $terms as $term ) {

					if ( 'uncategorized' === $term->slug ) {
						continue;
					}

					$get_tax[] = [
						'term_id'           => $term->term_id,
						'category_nicename' => $term->slug,
						'category_parent'   => $term->parent,
						'cat_name'          => $term->name,
						'taxonomy'          => $term->taxonomy,
					];
				}

				return ( count( $get_tax ) > 0 ) ? $get_tax : [];
			}

			$category = [];
			foreach ( $terms as $term ) {

				$post_terms = get_the_terms( $post_id, $term->taxonomy );

				if ( ! $post_terms ) {
					continue;
				}

				foreach ( $post_terms as $term ) {

					if ( empty( $term->slug ) ) {
						continue;
					}

					$category[] = [
						'domain'   => $term->taxonomy ?: '',
						'nicename' => $term->slug ?: '',
						'name'     => $term->name ?: '',
					];
				}
			}

			return ( count( $category ) > 0 ) ? $category : [];
		}

		private function get_quiz_pro_database_table( $type ) {
			global $wpdb;

			$table_name = "{$wpdb->prefix}learndash_pro_quiz_{$type}";

			if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) === $table_name ) {
				return $table_name;
			}

			return "{$wpdb->prefix}wp_pro_quiz_{$type}";
		}

		/**
		 * Undocumented function
		 *
		 * @return array
		 */
		protected function get_question_data( $question_ids = [] ) {
			global $wpdb;

			$question_table = $this->get_quiz_pro_database_table( 'question' );
			$category_table = $this->get_quiz_pro_database_table( 'category' );

			if ( ! empty( $question_ids ) ) {
				$question_ids    = array_filter(
					$question_ids,
					function ( $q_id ) {
						return ! is_null( $q_id );
					}
				);
				$count_questions = count( $question_ids );
				$placeholders    = array_fill( 0, $count_questions, '%d' );
				$format          = implode( ', ', $placeholders );
				$data            = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$question_table} WHERE id IN({$format})", $question_ids ) ); // db call ok; no-cache ok.
			} else {
				$data = $wpdb->get_results( "SELECT * FROM {$question_table}" ); // db call ok; no-cache ok.
			}
			$category_data = $wpdb->get_results( "SELECT * FROM {$category_table}" ); // db call ok; no-cache ok.

			if ( ! $data ) {
				return [
					'questions'  => [],
					'categories' => [],
				];
			}

			$questions  = [];
			$categories = [];
			foreach ( $data as $row ) {
				$questions[] = [
					'id'                                 => $row->id,
					'quiz_id'                            => $row->quiz_id,
					'online'                             => $row->online,
					'sort'                               => $row->sort,
					'title'                              => $row->title,
					'points'                             => $row->points,
					'question'                           => $row->question,
					'correct_msg'                        => $row->correct_msg,
					'incorrect_msg'                      => $row->incorrect_msg,
					'correct_same_text'                  => $row->correct_same_text,
					'tip_enabled'                        => $row->tip_enabled,
					'tip_msg'                            => $row->tip_msg,
					'answer_type'                        => $row->answer_type,
					'show_points_in_box'                 => $row->show_points_in_box,
					'answer_points_activated'            => $row->answer_points_activated,
					'answer_data'                        => addslashes( $row->answer_data ),
					'category_id'                        => $row->category_id,
					'answer_points_diff_modus_activated' => $row->answer_points_diff_modus_activated,
					'disable_correct'                    => $row->disable_correct,
					'matrix_sort_answer_criteria_width'  => $row->matrix_sort_answer_criteria_width,
				];
			}

			foreach ( $category_data as $row ) {
				$categories[] = [
					'id'   => $row->category_id,
					'name' => $row->category_name,
				];
			}

			return [
				'questions'  => array_chunk( $questions, $this->max_chunk ),
				'categories' => $categories,
			];
		}

		protected function export_media_files( $post_id, $post_type, $data ) {
			$upload_dir  = wp_upload_dir();
			$site_url    = site_url();
			$server_path = $upload_dir['basedir'];
			// REGEX:  http\://docker\.for\.mac\.localhost\:4000(?P<path>[^"\s]*?\.(png|jpg|jpeg|gif|svg|pdf))
			$site_regex = preg_quote( $site_url, '@' ) . '(?P<path>[^"\s]*?\.(png|jpg|jpeg|gif|svg|pdf))';

			$data['guid'] = str_replace( get_site_url(), self::SITE_URL, $data['guid'] );

			if ( 'sfwd-assignment' === $post_type ) {
				$file_link = str_replace( $site_url . '/wp-content/uploads', '', $data['meta']['file_link'][0] );
				$file_path = str_replace( $server_path, '', rawurldecode( $data['meta']['file_path'][0] ) );

				$this->media->add_files(
					[
						$file_link,
						$file_path,
					]
				);

				$data['meta']['file_link'][0] = self::SITE_URL . '/wp-content/uploads' . $file_link;
				$data['meta']['file_path'][0] = rawurlencode( self::SERVER_PATH . $file_path );
			}

			if ( 'sfwd-courses' === $post_type ) {
				$unserialized_data = unserialize( $data['meta']['_sfwd-courses'][0] );
				if ( isset( $unserialized_data['sfwd-courses_course_materials'] ) ) {
					$content = $unserialized_data['sfwd-courses_course_materials'];

					$unserialized_data['sfwd-courses_course_materials'] = $this->replace_matches_from_content( $site_regex, $site_url, $content );

					$data['meta']['_sfwd-courses'][0] = serialize( $unserialized_data );
				}
			}

			if ( 'sfwd-lessons' === $post_type ) {
				$unserialized_data = unserialize( $data['meta']['_sfwd-lessons'][0] );
				if ( isset( $unserialized_data['sfwd-lessons_lesson_materials'] ) ) {
					$content = $unserialized_data['sfwd-lessons_lesson_materials'];

					$unserialized_data['sfwd-lessons_lesson_materials'] = $this->replace_matches_from_content( $site_regex, $site_url, $content );

					$data['meta']['_sfwd-lessons'][0] = serialize( $unserialized_data );
				}
			}

			if ( 'sfwd-topic' === $post_type ) {
				$unserialized_data = unserialize( $data['meta']['_sfwd-topic'][0] );
				if ( isset( $unserialized_data['sfwd-topic_materials'] ) ) {

					$content = $unserialized_data['sfwd-topic_materials'];

					$unserialized_data['sfwd-topic_materials'] = $this->replace_matches_from_content( $site_regex, $site_url, $content );

					$data['meta']['_sfwd-topic'][0] = serialize( $unserialized_data );
				}
			}

			if ( 'sfwd-quiz' === $post_type ) {
				$unserialized_data = unserialize( $data['meta']['_sfwd-quiz'][0] );
				if ( isset( $unserialized_data['sfwd-quiz_materials'] ) ) {
					$content = $unserialized_data['sfwd-quiz_materials'];

					$unserialized_data['sfwd-quiz_materials'] = $this->replace_matches_from_content( $site_regex, $site_url, $content );

					$data['meta']['_sfwd-quiz'][0] = serialize( $unserialized_data );
				}
			}

			$data['post_content'] = $this->replace_matches_from_content( $site_regex, $site_url, $data['post_content'] );

			if ( isset( $data['meta']['_thumbnail_id'] ) ) {
				$this->export_attachments( $data['meta']['_thumbnail_id'][0] );
			}

			return $data;
		}

		private function export_attachments( $thumbnail_id ) {
			if ( empty( $thumbnail_id ) ) {
				return;
			}
			$upload_dir  = wp_upload_dir();
			$site_url    = $upload_dir['baseurl'];
			$server_path = $upload_dir['basedir'];

			$post         = get_post( $thumbnail_id, ARRAY_A );
			$old_guid     = $post['guid'];
			$post['guid'] = str_replace( get_site_url(), self::SITE_URL, $post['guid'] );

			$terms = $this->get_terms( $post['ID'] );
			if ( $terms ) {
				$post['category'] = $terms;
			}

			$meta = get_post_meta( $post['ID'] );
			if ( count( $meta ) > 0 ) {

				if ( isset( $meta['_wp_attached_file'] ) ) {
					$image = $meta['_wp_attached_file'][0];
					$image = preg_replace( '@(.+)(\..+$)@', '$1*', $image );
					$this->media->add_file( $image );
				}
				$post['meta'] = $meta;
			}

			$this->attachments_posts[ $post['ID'] ] = $post;

			$to_download = str_replace( $site_url, '', $old_guid );
			$this->media->add_file( $to_download );
		}

		private function replace_matches_from_content( $regex, $site_url, $content ) {
			if ( preg_match_all( "@{$regex}@", $content, $matches ) ) {
				foreach ( $matches['path'] as $match ) {
					$match = str_replace( 'wp-content/uploads', '', $match );
					$this->media->add_file( $match );
				}
				$content = str_replace( $site_url, self::SITE_URL, $content );
			}

			return $content;
		}

		public function get_users_meta_data( $user ) {
			$export_usermeta = [];
			$user_metas      = (array) get_user_meta( $user->ID, '', true );
			$user_metas      = array_filter(
				array_map(
					function( $a ) {
						return $a[0];
					},
					$user_metas
				)
			);

			foreach ( $user_metas as $k => $v ) {

				switch ( $k ) {
					case 'nickname':
					case 'first_name':
					case 'last_name':
					case 'rich_editing':
					case 'syntax_highlighting':
					case 'comment_shortcuts':
					case 'show_admin_bar_front':
					case 'admin_color':
					case 'wp_capabilities':
					case 'learndash-last-login':
					case '_sfwd-course_progress':
					case 'learndash_group_leaders_' === preg_replace( '/\d+/u', '', $k ):
					case 'learndash_group_users_' === preg_replace( '/\d+/u', '', $k ):
					case 'course_completed_' === preg_replace( '/\d+/u', '', $k ):
					case 'course__access_from' === preg_replace( '/\d+/u', '', $k ):
					case '_sfwd-quizzes':
					case 'ld_assignment_message':
					case 'course_points':
					case 'ld_register_form':
					case '_ld_registered_post':
					case '_ld_lostpassword_redirect_to':
					case 'completed_' === preg_replace( '/\d+/u', '', $k ):
					case 'learndash_course_expired_' === preg_replace( '/\d+/u', '', $k ):
						$export_usermeta[ $k ] = $v;
						break;

					default:
						break;
				}
			}
			return $export_usermeta;
		}

		/**
		 * Users data.
		 *
		 * @return array
		 */
		public function get_users_data( $opencourse_data ) {
			$users = get_users();

			$all_users = [];

			foreach ( $users as $key => $user ) {

				if ( function_exists( 'learndash_user_get_enrolled_courses' ) && ! empty( $opencourse_data ) ) {
					$users_courses = learndash_user_get_enrolled_courses( $user->ID );
				} elseif ( function_exists( 'learndash_get_user_courses_from_meta' ) ) {
					$users_courses = learndash_get_user_courses_from_meta( $user->ID );
				} else {
					$users_courses = '';
				}

				if ( empty( $users_courses ) ) {
					continue;
				}

				$all_users[ $key ] = (array) $user->data;

				$export_usermeta = $this->get_users_meta_data( $user );

				$all_users[ $key ]['meta'] = $export_usermeta;
			}

			return array_chunk( $all_users, $this->max_chunk, true );
		}
	}
}
