<?php
/**
 * LDIE import export class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Export_File' ) ) {
	/**
	 * Export class
	 */
	class Ld_Admin_Export_File {
		/**
		 * Undocumented variable
		 *
		 * @var string
		 */
		private $text = null;


		/**
		 * Construct
		 */
		public function __construct() {
			// trigger import via ajax!
			$this->text = Ld_Text::text( 'notices' );

			add_action( 'wp_ajax_export_ld_data', [ $this, 'export_ld_data' ] );
			add_action( 'wp_ajax_ldie_export_get_files', [ $this, 'get_exported_files' ] );
			add_action( 'wp_ajax_ldie_export_delete_files', [ $this, 'delete_exported_files' ] );
		}


		public function get_exported_files() {
			try {
				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$files = Ld_Admin_File_System::get_exported_files();
				echo wp_json_encode(
					[
						'status' => 'success',
						'data'   => $files,
					]
				);
			} catch ( \Exception $e ) {
				wp_send_json_error(
					[
						'status'  => 'error',
						'message' => $e->getMessage(),
					],
					500
				);

			}
			exit();
		}

		public function delete_exported_files() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_export_cleanup' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$specific_dir = isset( $_POST['folder'] ) ? sanitize_text_field( wp_unslash( $_POST['folder'] ) ) : '';

				Ld_Admin_File_System::delete_export_dir( $specific_dir );

				echo wp_json_encode(
					[
						'status' => 'success',
					]
				);
			} catch ( \Exception $e ) {
				wp_send_json_error(
					[
						'status'  => 'error',
						'message' => $e->getMessage(),
					],
					500
				);

			}
			exit();

		}

		/**
		 * Function: export_ld_data
		 * call back action from ajax request to export the data
		 *
		 * @since 1.0.0
		 * @throws \Exception When the data is invalid.
		 * @return void
		 */
		public function export_ld_data() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_export_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$post_types             = isset( $_POST['post_types'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['post_types'] ) ) ) : [];
				$data_type              = isset( $_POST['data_type'] ) ? sanitize_text_field( wp_unslash( $_POST['data_type'] ) ) : ''; // JSON/XML.
				$export_type            = isset( $_POST['export_type'] ) ? sanitize_text_field( wp_unslash( $_POST['export_type'] ) ) : '';
				$users_data             = isset( $_POST['users_data'] ) ? sanitize_text_field( wp_unslash( $_POST['users_data'] ) ) : '';
				$course_ids             = isset( $_POST['post_ids'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['post_ids'] ) ) ) : [];
				$max_chunk              = isset( $_POST['max_chunk'] ) ? intval( sanitize_text_field( wp_unslash( $_POST['max_chunk'] ) ) ) : 150;
				$content_global_setting = isset( $_POST['content_global_setting'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['content_global_setting'] ) ) ) : [];
				$opencourse_checkboxes  = isset( $_POST['opencourse_data'] ) ? sanitize_text_field( wp_unslash( $_POST['opencourse_data'] ) ) : '';

				if ( 'per-course' === $export_type ) {
					if ( empty( $course_ids ) ) {
						throw new \Exception( __( 'Select the Courses to export', 'ea_import_export' ) );
					}

					$export_data_handler = new Ld_Admin_Export_File_Per_Course( $data_type, $export_type, $max_chunk );
				} else {
					$export_data_handler = new Ld_Admin_Export_File_All_Data( $data_type, $export_type, $max_chunk );
				}

				$response = $export_data_handler->export_data( $post_types, $course_ids, $content_global_setting, $users_data, $opencourse_checkboxes );

				echo wp_json_encode(
					[
						'status'      => $this->text['status'][1],
						'message'     => 'Successfully exported',
						'file_status' => $response['success'],
						'file_source' => $response['filename'],
					]
				);
			} catch ( \Exception $e ) {
				wp_send_json_error(
					[
						'status'  => 'error',
						'message' => $e->getMessage(),
					],
					500
				);
			}

			exit();
		}

	}
}
