<?php
/**
 * LDIE import export class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

use LDIE\Admin\Ld_Admin_Global_Content_Setting;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Export_JSON_File' ) ) {

	/**
	 * Export class
	 */
	class Ld_Admin_Export_JSON_File {



		/**
		 * Format index for json
		 *
		 * @param string $index Index.
		 * @return string
		 */
		private function format_index( $index = null ) {
			if ( ! $index ) {
				return;
			}

			$toreformat = [
				'ID'          => 'wp_post_id',
				'post_author' => 'wp_author_id',
			];

			return isset( $toreformat[ $index ] ) ? $toreformat[ $index ] : 'wp_' . $index;
		}

		/**
		 * Undocumented function
		 *
		 * @param array $data data.
		 * @return string
		 */
		public function generate_data( $data = null ) {
			if ( ! $data ) {
				return;
			}
			$json          = [];
			$json['keyid'] = md5( 'ldie' );

			foreach ( $data as $id => $row ) {
				if ( 'categories' === $id ) {
					$category_index          = $this->format_index( 'categories' );
					$json[ $category_index ] = $this->format_categories( $row );
					continue;
				}

				if ( 'quiz_pro' === $id && $row ) {
					$json['quiz_pro'] = $this->format_quiz_pro( $row );
					continue;
				}

				$item = [];
				foreach ( $row as $index => $val ) {
					if ( 'category' === $index ) {
						list($child_categories, $ld_categories) = $this->format_ld_categories( $val );
						$item['categories']                     = $child_categories;
						$item['ld_categories']                  = $ld_categories;
						continue;
					}

					if ( 'meta' === $index ) {
						$formatted          = $this->format_index( 'postmeta' );
						$item[ $formatted ] = $this->format_ld_meta( $val );
						continue;
					}

					if ( 'guid' === $index ) {
						$item[ $index ] = [
							'is_permalink' => false,
							'value'        => $val,
						];
						continue;
					}

					$index_formatted          = $this->format_index( $index );
					$item[ $index_formatted ] = $val;
				}

				$json['wp_data'][] = $item;
			}

			return wp_json_encode( $json );
		}


		/**
		 * Prepare the Categories/Taxonomies for the JSON file
		 *
		 * @param array $raw_categories .
		 * @return array
		 */
		private function format_categories( $raw_categories ) {
			$formatted = [];

			foreach ( $raw_categories as $index => $category ) {
				$id_formatted            = $this->format_index( 'term_id' );
				$taxonomy_formatted      = $this->format_index( 'term_taxonomy' );
				$nicename_formatted      = $this->format_index( 'category_nicename' );
				$parent_formatted        = $this->format_index( 'category_parent' );
				$category_name_formatted = $this->format_index( 'category_name' );

				if ( empty( $category['category_nicename'] ) ) {
					continue;
				}

				if ( array_search( $category['category_nicename'], array_column( $formatted, $nicename_formatted ), true ) ) {
					continue;
				}

				$formatted[] = [
					$id_formatted            => $category['term_id'] ?: '',
					$taxonomy_formatted      => $category['taxonomy'] ?: '',
					$nicename_formatted      => $category['category_nicename'] ?: '',
					$parent_formatted        => $category['category_parent'] ?: '',
					$category_name_formatted => $category['cat_name'] ?: '',
				];
			}

			return $formatted;
		}

		/**
		 * Prepare the LeadnDash Categories for the JSON file
		 *
		 * @param array $ld_categories .
		 * @return array
		 */
		private function format_ld_categories( $ld_categories ) {
			$formatted          = [];
			$category_formatted = [];

			foreach ( $ld_categories as $index => $category ) {

				if ( array_search( $category['nicename'], array_column( $category_formatted, 'nicename' ), true ) ) {
					continue;
				}

				$category_formatted[] = [
					'domain'   => $category['domain'],
					'nicename' => $category['nicename'],
					'name'     => $category['name'],
				];

				$formatted[] = [
					'ld_slug'          => $category['nicename'],
					'ld_taxonomy'      => $category['domain'],
					'ld_category_name' => $category['name'],
				];
			}

			return [
				$category_formatted,
				$formatted,
			];
		}


		/**
		 * Prepare the LearnDash Metas for the JSON file
		 *
		 * @param array $ld_metas .
		 * @return array
		 */
		private function format_ld_meta( $ld_metas ) {
			$formatted = [];

			foreach ( $ld_metas as $key => $value ) {
				$formatted_key   = $this->format_index( 'meta_key' );
				$formatted_value = $this->format_index( 'meta_value' );

				$formatted[] = [
					$formatted_key   => $key,
					$formatted_value => $value[0],
				];
			}

			return $formatted;
		}

		/**
		 * Prepare Quiz Pro for JSON file
		 *
		 * @param array $quiz .
		 * @return array
		 */
		private function format_quiz_pro( $quiz ) {
			$formatted = [];
			$questions = [];

			$questions_raw = $quiz['questions'];
			if ( $questions_raw ) {
				foreach ( $questions_raw as $question ) {
					$questions[] = preg_replace( "/\r|\n/", ' ', serialize( $question ) );
				}
			}

			$formatted = [
				'questions'  => $questions,
				'categories' => $quiz['categories'],
			];

			return $formatted;
		}

		public function global_content_setting( $content_global_settings ) {
			$global_settings = array();

			if ( empty( $content_global_settings ) ) {
				return $global_settings;
			}

			$settings = array();
			foreach ( $content_global_settings as $content_global_setting ) {

				switch ( $content_global_setting ) {
					case 'sfwd-courses-setting':
						$settings['courses_setting'] = Ld_Admin_Global_Content_Setting::global_course_settings();
						break;

					case 'sfwd-lessons-setting':
						$settings['lesson_setting'] = Ld_Admin_Global_Content_Setting::global_lesson_settings();
						break;

					case 'sfwd-topic-setting':
						$settings['topic_setting'] = Ld_Admin_Global_Content_Setting::global_topic_settings();
						break;

					case 'sfwd-quiz-setting':
						$settings['quizzes_setting'] = Ld_Admin_Global_Content_Setting::global_quizzes_settings();
						break;

					case 'sfwd-question-setting':
						$settings['questions_setting'] = Ld_Admin_Global_Content_Setting::global_questions_settings();
						break;

					case 'sfwd-certificates-setting':
						$settings['certificates_setting'] = Ld_Admin_Global_Content_Setting::global_certificates_settings();
						break;
					case 'groups-setting':
						$settings['groups_setting'] = Ld_Admin_Global_Content_Setting::global_groups_settings();
						break;

					case 'sfwd-assignment-setting':
						$settings['assignments_setting'] = Ld_Admin_Global_Content_Setting::global_assignment_settings();
						break;

					default:
						break;
				}
			}

			$global_settings['keyid'] = md5( 'ldie' );

			$global_settings['ld_global_setting'] = $settings;

			return wp_json_encode( $global_settings );
		}

		/**
		 * Users data.
		 *
		 * @return array
		 */
		public function generate_users_data( $users ) {
			$json          = [];
			$json['keyid'] = md5( 'ldie' );

			$user_index          = $this->format_index( 'users_data' );
			$json[ $user_index ] = $users;

			return wp_json_encode( $json );
		}

		public function generate_categories_data( $categories ) {

			$json          = [];
			$json['keyid'] = md5( 'ldie' );

			$category_index          = $this->format_index( 'categories' );
			$json[ $category_index ] = $this->format_categories( $categories );

			return wp_json_encode( $json );
		}

		public function generate_quiz_pro_data( $data ) {
			$json          = [];
			$json['keyid'] = md5( 'ldie' );

			$json['quiz_pro'] = $this->format_quiz_pro( $data );

			return wp_json_encode( $json );
		}
	}
}
