<?php
/**
 * LDIE import export class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * @author Jonatan Treviño <galtrev@gmail.com>
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_File_System' ) ) {

	/**
	 * Export class
	 */
	class Ld_Admin_File_System {


		/**
		 * Directory to assign the created file
		 *
		 * @var string $dir
		 */
		private $dir;

		/**
		 * Extension of the file
		 *
		 * @var string $extension
		 */
		private $extension;

		/**
		 * Prefix of the filename
		 *
		 * @var string $prefix
		 */
		private $prefix;

		/**
		 * Check if will create a zip file or not
		 *
		 * @var boolean $is_zip
		 */
		private $is_zip = false;

		/**
		 * ..
		 *
		 * @var boolean $is_zip
		 */
		private $is_export = true;

		/**
		 * Check if will create a zip file or not
		 *
		 * @var string $date_file
		 */
		private $date_file;

		/**
		 * Construct
		 *
		 * @param string  $dir default '' .
		 * @param string  $extension default xml .
		 * @param string  $prefix File name prefix .
		 * @param boolean $is_zip .
		 * @param boolean $is_export .
		 */
		public function __construct( $dir = '', $extension = 'xml', $prefix = '', $is_zip = false, $is_export = true ) {
			$this->dir       = $dir;
			$this->extension = $extension;
			$this->prefix    = $prefix;
			$this->is_zip    = $is_zip;
			$this->is_export = $is_export;
			$this->date_file = gmdate( 'Y-m-d-H-i-s' );
			$this->generate_folders();
		}

		/**
		 * Generate Folders if don't exists
		 *
		 * @return void
		 * @throws \Exception When Filesystem cannot be initialized.
		 */
		private function generate_folders() {
			try {
				\WP_Filesystem();
				global $wp_filesystem;

				$upload_dir = $this->get_upload_folder();

				if ( ! $wp_filesystem->exists( $upload_dir ) ) {
					$folders = $this->get_upload_folder();
					$str     = $wp_filesystem->wp_content_dir();
					$folders = str_replace( $str, '', $folders );
					$folders = explode( '/', $folders );

					foreach ( $folders as $folder ) {
						$str .= "/{$folder}";
						if ( ! $wp_filesystem->exists( $str ) ) {
							$wp_filesystem->mkdir( $str );
						}
					}
				}
			} catch ( \Exception $error ) {
				throw new Exception( __( 'Cannot initialize filesystem', 'ea_import_export' ) );
			}

		}

		/**
		 * Given a path will search into wp-uploads/*, if exists will copy the file and return the new path
		 *
		 * @param string $file .
		 */
		public function copy_file_to_export( $file ) {
			global $wp_filesystem;
			WP_Filesystem();
			$file          = untrailingslashit( $file );
			$upload_folder = $wp_filesystem->wp_content_dir() . 'uploads/';
			$folder        = $this->get_upload_folder() . '/media/';
			$upload_file   = $upload_folder . $file;
			$is_for_glob   = preg_match( '@\*$@', $file );

			if ( ! preg_match( '@\.(png|jpg|jpeg|gif|svg|pdf)$@', $file ) && ! $is_for_glob ) {
				return;
			}

			if ( ! \file_exists( $upload_file ) && ! $is_for_glob ) {
				return;
			}

			$files_from_glob = [];
			if ( $is_for_glob ) {
				$files = glob( $upload_file );
				foreach ( $files as $file_name ) {

					$dest              = str_replace( $upload_folder, $folder, $file_name );
					$files_from_glob[] = [
						'source' => $file_name,
						'dest'   => $dest,
					];
				}

				if ( empty( $files_from_glob ) ) {
					return;
				}
			}

			if ( ! $wp_filesystem->exists( $folder ) ) {
				$wp_filesystem->mkdir( $folder );
			}

			$folders_file = preg_replace( '@/[^/]+.\w+$@', '', $folder . $file );
			wp_mkdir_p( $folders_file );
			$folder_file = $folder . $file;

			foreach ( $files_from_glob as $file_to_export ) {
				if ( ! is_dir( $file_to_export['source'] ) ) {
					copy( $file_to_export['source'], $file_to_export['dest'] );
				}
			}

			if ( \file_exists( $upload_file ) && ! is_dir( $upload_file ) ) {
				copy( $upload_file, $folder_file );
			}
		}

		/**
		 * Check if file exists
		 *
		 * @param string $file .
		 * @return boolean
		 */
		public static function exists_file( $file ) {
			global $wp_filesystem;

			return $wp_filesystem->exists( $file );
		}

		public static function get_exported_files() {
			WP_Filesystem();
			global $wp_filesystem;
			$server_path        = $wp_filesystem->wp_content_dir() . 'uploads/';
			$exported_files_dir = 'honorswp/export';

			if ( ! $wp_filesystem->exists( $server_path . $exported_files_dir ) ) {
				return [];
			}

			$raw_files = array_values( array_diff( scandir( $server_path . $exported_files_dir, 1 ), [ '..', '.', '.DS_Store' ] ) );

			$raw_files = array_filter(
				$raw_files,
				function ( $file ) {
					return ( ! preg_match( '@\.zip$@', $file ) && ! preg_match( '@^export$@', $file ) );
				}
			);

			$baseurl = wp_get_upload_dir()['baseurl'];
			$files   = [];

			foreach ( $raw_files as $file ) {
				$zip_path = '';
				if ( file_exists( $server_path . $exported_files_dir . '/' . $file . '.zip' ) ) {
					$zip_path = $baseurl . '/' . $exported_files_dir . '/' . $file . '.zip';
				}
				$files[] = [
					'folder_path' => 'uploads/' . $exported_files_dir . '/' . $file,
					'zip_path'    => $zip_path,
				];
			}

			return $files;
		}

		public static function get_imported_directories( $dir = '' ) {
			WP_Filesystem();
			global $wp_filesystem;
			$server_path        = $wp_filesystem->wp_content_dir() . 'uploads/';
			$exported_files_dir = untrailingslashit( 'honorswp/import/' . $dir );
			$full_path          = $server_path . $exported_files_dir;

			if ( ! $wp_filesystem->exists( $full_path ) || ! is_dir( $full_path ) ) {
				return [];
			}

			$raw_files = array_values( array_diff( scandir( $full_path, 1 ), [ '..', '.', '.DS_Store' ] ) );

			$raw_files = array_filter(
				$raw_files,
				function ( $file ) {
					return ( ! preg_match( '@\.zip$@', $file ) );
				}
			);

			return array_values( $raw_files );
		}

		public static function scan_import_directory( $dir ) {
			WP_Filesystem();
			global $wp_filesystem;
			$server_path        = $wp_filesystem->wp_content_dir() . 'uploads/';
			$exported_files_dir = untrailingslashit( 'honorswp/import/' . $dir );
			$full_path          = $server_path . $exported_files_dir;

			$files = self::get_imported_directories( $dir );

			$media_folder = $full_path . '/media';

			if ( is_dir( $media_folder ) ) {
				$wp_upload_folder = $wp_filesystem->wp_content_dir() . 'uploads/';
				copy_dir( $media_folder, $wp_upload_folder );
			}

			$files        = array_filter(
				$files,
				function ( $file ) {
					return preg_match( '@\.(json|xml)$@', $file );
				}
			);
			$sorted_files = self::sort_zip_files( array_values( $files ) );
			return [
				'files' => $sorted_files,
				'dir'   => $dir,
			];
		}


		public function generate_export_files() {
			try {
				$url   = $this->get_upload_folder( false );
				$files = $this->get_files();

				if ( $this->get_is_zip() || is_array( $files ) ) {
					$zip_created = $this->create_zip_file();

					if ( ! is_wp_error( $zip_created ) ) {
						return [
							'success'  => true,
							'filename' => $zip_created ? untrailingslashit( $url ) . '.zip' : '',
						];
					}
				}

				return [
					'success'  => true,
					'filename' => trailingslashit( $url ) . $files,
				];
			} catch ( \Exception $error ) {
				throw new \Exception( __( 'There is no data to be exported. ', 'ea_import_export' ) . $error->getMessage() );
			}
		}

		private function get_files() {
			$folder          = $this->get_upload_folder();
			$files_in_folder = array_values( array_diff( scandir( $folder ), [ '..', '.' ] ) );

			if ( empty( $files_in_folder ) ) {
				throw new \Exception( __( 'No files to be exported', 'ea_import_export' ) );
			}

			if ( 1 === count( $files_in_folder ) ) {
				if ( ! file_exists( $folder . '/' . $files_in_folder[0] ) ) {
					throw new \Exception( __( 'File does not exist', 'ea_import_export' ) );
				}

				return $files_in_folder[0];
			}

			return $files_in_folder;
		}

		/**
		 * Create file based on extension.
		 *
		 * @param string $content Content of the File .
		 * @param string $prefix New folder name if needed .
		 * @return string
		 */
		public function create_file( $content, $prefix = '' ) {
			try {
				WP_Filesystem();
				global $wp_filesystem;

				$upload_dir = $this->get_upload_folder();
				$dir        = $wp_filesystem->find_folder( $upload_dir );
				$filename   = $this->generate_file_name( $prefix );
				$file       = trailingslashit( $dir ) . $filename;
				$success    = $wp_filesystem->put_contents( $file, $content, FS_CHMOD_FILE );

				return $success;
			} catch ( \Exception $error ) {
				return new \WP_Error( 'filesystem_error', __( 'Cannot initialize filesystem', 'ea_import_export' ) );
			}
		}

		public function create_files( $content, $prefix = '' ) {
			if ( ! is_array( $content ) ) {
				return $this->create_file( $content, $prefix );
			}

			if ( 0 === count( $content ) ) {
				return false;
			}

			if ( is_array( $content ) && 1 === count( $content ) ) {
				return $this->create_file( $content[0], $prefix );
			}

			if ( ! $this->is_zip ) {
				$this->set_is_zip( true );
			}

			$pad_start = mb_strlen( (string) count( $content ) );
			foreach ( $content as $key => $value ) {
				$index = str_pad( $key + 1, $pad_start, '0', STR_PAD_LEFT );
				$this->create_file( $value, "{$prefix}_{$index}" );
			}

			return true;
		}

		/**
		 * Get Upload Folder
		 *
		 * @param boolean $is_server if is going to be an URL (false) or if is a server path (true).
		 * @return string
		 */
		private function get_upload_folder( $is_server = true ) {
			global $wp_filesystem;
			$upload_dir = "honorswp/{$this->dir}";

			if ( $this->is_export ) {
				$date        = $this->date_file;
				$upload_dir .= "/export_{$date}_{$this->extension}";
			}

			$baseurl = wp_get_upload_dir()['baseurl'];

			return $is_server ? $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( $upload_dir ) : trailingslashit( "{$baseurl}/{$upload_dir}" );
		}

		/**
		 * Generate a File name based on date and time
		 *
		 * @param string $prefix name to include in the file.
		 * @return string
		 */
		private function generate_file_name( $prefix = '' ) {
			$prefix = empty( $prefix ) ? $this->prefix : $prefix;

			$date     = $this->date_file;
			$filename = str_replace( '--', '', "{$this->dir}-file-{$prefix}-{$date}.{$this->extension}" );

			return $filename;
		}


		private function create_zip_file() {
			$dir = $this->get_upload_folder();
			if ( ! class_exists( 'ZipArchive' ) ) {
				return new \WP_Error( 'zip_class_error', esc_html__( 'Can not create zip file', 'ea_import_export' ) );
			}
			$zip          = new \ZipArchive();
			$zip_filename = rtrim( $dir, '/' ) . '.zip';

			if ( $zip->open( $zip_filename, \ZipArchive::CREATE ) !== true ) {
				throw new \Exception( esc_html__( 'Error creating Zip File', 'ea_import_export' ) );
			}

			$this->create_zip_subdirectories( $dir, $zip );

			if ( \ZipArchive::ER_OK !== $zip->status ) {
				throw new \Exception( esc_html__( 'Failed to write files to Zip', 'ea_import_export' ) );
			}
			$zip->close();


			return true;
		}

		private function create_zip_subdirectories( $source, $zip ) {

			if ( empty( $source ) ) {
				return;
			}

			$source = realpath( $source );

			if ( is_file( $source ) ) {
				$zip->addFile( $source, basename( $source ) );
			}

			if ( is_dir( $source ) ) {
				$iterator = new \RecursiveDirectoryIterator( $source );
				$iterator->setFlags( \RecursiveDirectoryIterator::SKIP_DOTS );
				$files_iterator = new \RecursiveIteratorIterator( $iterator, \RecursiveIteratorIterator::SELF_FIRST );

				foreach( $files_iterator as $file ) {
					$file = realpath( $file );

					if ( is_dir( $file ) ) {
						$zip->addEmptyDir( str_replace( $source . DIRECTORY_SEPARATOR, '', $file . DIRECTORY_SEPARATOR ) );
					} elseif( is_file( $file ) ) {
						$zip->addFile( $file, str_replace( $source . DIRECTORY_SEPARATOR, '', $file ) );
					}
				}
			}
		}

		public static function decompress_zip_file( $filepath, $filename ) {
			global $wp_filesystem;

			$import_dir = str_replace( '.zip', '', $filename );
			WP_Filesystem();
			$upload_dir = $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( 'honorswp/import' );

			if ( ! is_dir( $upload_dir ) ) {
				mkdir( $upload_dir, 0777, true );
			}

			$files_dir = "{$upload_dir}/{$import_dir}/";

			if ( ! is_dir( $files_dir ) ) {
				mkdir( $files_dir, 0777, true );
			}

			// WP file function.
			$unzipfile = unzip_file( $filepath, $files_dir );
			if ( is_wp_error( $unzipfile ) ) {
				throw new \Exception( $unzipfile->get_error_message() );
			}

			$raw_files = array_values( array_diff( scandir( $files_dir ), [ '..', '.', 'media' ] ) );

			if ( 1 === count( $raw_files ) && is_dir( $files_dir . $raw_files[0] ) ) {
				copy_dir( $files_dir . $raw_files[0], $files_dir );
				$wp_filesystem->rmdir( $files_dir . $raw_files[0], true );
				$raw_files = array_values( array_diff( scandir( $files_dir ), [ '..', '.', 'media' ] ) );
			}

			$media_folder = $files_dir . 'media';
			$files        = [];

			if ( is_dir( $media_folder ) ) {
				$wp_upload_folder = $wp_filesystem->wp_content_dir() . 'uploads/';
				copy_dir( $media_folder, $wp_upload_folder );
			}

			foreach ( $raw_files as $file ) {
				if ( preg_match( '/\.(json|xml)$/', $file, $matches ) && 'cache.json' !== $file ) {
					$files[] = $file;
				}
			}

			$sorted_files = self::sort_zip_files( $files );

			return [
				'files' => $sorted_files,
				'dir'   => $import_dir,
			];

		}

		private static function sort_zip_files( $files ) {
			$sorted = [];

			// First Categories.
			$categories = preg_grep( '/wp_categories/', $files );

			if ( $categories ) {
				$sorted = array_merge( $sorted, $categories );
			}

			// Groups, Certificates, Assignments.
			$normal = preg_grep( '/(groups|certificates|assignment)/', $files );
			if ( $normal ) {
				$sorted = array_merge( $sorted, $normal );
			}

			// Courses.
			$courses = preg_grep( '/courses/', $files );
			if ( $courses ) {
				sort( $courses );

				$sorted = array_merge( $sorted, $courses );
			}

			// Quiz Data.
			$quizzes_questions = preg_grep( '/quiz_pro/', $files );
			if ( $quizzes_questions ) {
				if ( 1 < count( $quizzes_questions ) ) {
					sort( $quizzes_questions );
				}

				$sorted = array_merge( $sorted, $quizzes_questions );
			}

			// Quiz Data.
			$quizzes = preg_grep( '/sfwd-quiz/', $files );
			if ( $quizzes ) {
				if ( 1 < count( $quizzes ) ) {
					sort( $quizzes );
				}

				$sorted = array_merge( $sorted, $quizzes );
			}

			// Question.
			$questions = preg_grep( '/question/', $files );
			if ( $questions ) {
				sort( $questions );

				$sorted = array_merge( $sorted, $questions );
			}

			// Lesson.
			$lessons = preg_grep( '/lessons/', $files );
			if ( $lessons ) {
				sort( $lessons );

				$sorted = array_merge( $sorted, $lessons );
			}

			// Topic.
			$topic = preg_grep( '/topic/', $files );
			if ( $topic ) {
				sort( $topic );

				$sorted = array_merge( $sorted, $topic );
			}

			return array_unique( array_merge( $sorted, $files ) );
		}

		public static function get_wp_upload_dir( $filepath ) {
			global $wp_filesystem;
			WP_Filesystem();
			return $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( "honorswp/import/{$filepath}" );
		}

		public static function upload_file( $filepath, $filename, $extension ) {
			global $wp_filesystem;
			$import_dir = str_replace( '/.(json|xml)/', '', $filename );
			WP_Filesystem();
			$upload_dir = $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( "honorswp/import/{$import_dir}" );

			if ( ! is_dir( $upload_dir ) ) {
				mkdir( $upload_dir, 0777, true );
			}

			$uploaded_file = trailingslashit( $upload_dir ) . $filename;
			copy( $filepath, $uploaded_file );

			return [
				'files' => [ $filename ],
				'dir'   => $import_dir,
			];
		}

		public static function delete_import_dir( $dir ) {
			global $wp_filesystem;
			WP_Filesystem();
			$import_dir = $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( "honorswp/import/{$dir}" );
			$wp_filesystem->rmdir( $import_dir, true );
		}

		public static function delete_export_dir( $dir ) {
			global $wp_filesystem;
			WP_Filesystem();
			$export_dir = $wp_filesystem->wp_content_dir() . 'uploads/' . untrailingslashit( "honorswp/export/{$dir}" );
			$wp_filesystem->rmdir( $export_dir, true );

			if ( ! empty( $dir ) && file_exists( $export_dir . 'zip' ) ) {
				$wp_filesystem->delete( $export_dir . 'zip' );
			}
		}



		public function set_extension( $ext ) {
			$this->extension = $ext;
		}

		public function get_extension() {
			return $this->extension;
		}

		public function set_is_zip( $zip ) {
			$this->is_zip = $zip;
		}

		public function get_is_zip() {
			return $this->is_zip;
		}
	}
}
