<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_Abstract_File' ) ) {
	/**
	 * Admin Import
	 */
	abstract class Ld_Admin_Import_Abstract_File extends Ld_Admin_Import_Common_File {

		/**
		 * Abstract
		 *
		 * @param string $filepath .
		 * @return mixed
		 */
		abstract protected function prepare_file_data( $filepath );

		/**
		 * Abstract
		 *
		 * @param array $data .
		 * @return void
		 */
		abstract protected function is_valid_data( $data );

		/**
		 * Abstract function to initialize the structure
		 *
		 * @param array $data .
		 * @return void
		 */
		abstract protected function init( $data );

		/**
		 * Abstract
		 *
		 * @param array $categories .
		 * @return void
		 */
		abstract protected function import_categories( $categories );

		/**
		 * Abstract
		 *
		 * @param array $posts .
		 */
		abstract protected function import_posts( $posts );

		/**
		 * Abstract
		 *
		 * @param array $settings .
		 */
		abstract protected function import_global_content_setting( $settings );

		/**
		 * Abstract
		 */
		abstract protected function import_users_data( $users_data );

		/**
		 * Count data passing a file name
		 *
		 * @param string $filename .
		 * @param string $key .
		 *
		 * @return int
		 */
		public function count_data_from_file( $filename, $key ) {
			$data = $this->prepare_file_data( $filename );
			$data = $this->init( $data );

			return $this->count_data( $data, $key );
		}

		public function count_data( $data, $key ) {
			if ( 'posts' === $key && ! is_array( $data['posts'] ) ) {
				$data['posts'] = [ $data['posts'] ];
			}

			$total_count = isset( $data[ $key ] ) ? count( $data[ $key ] ) : 0;

			if ( 'quiz_pro' === $key ) {
				$total_count = count( $data[ $key ]['questions'] );
			}

			return $total_count;
		}

		/**
		 * Do import
		 *
		 * @param string $filename .
		 * @throws \Exception When the data is invalid .
		 * @return array|null
		 */
		public function run( $filename, $index_to_import = null, $batch_to_import = null ) {
			$data = $this->prepare_file_data( $filename );

			return $this->do_imports( $data, $index_to_import, $batch_to_import );
		}

		/**
		 * Do Imports
		 *
		 * @param array $data .
		 * @return boolean
		 */
		protected function do_imports( $data, $index_to_import = null, $batch_to_import = null ) {
			try {
				$init_data            = $this->init( $data );
				$quiz_pro_total_count = 0;
				$post_total_count     = 0;

				if ( ! empty( $init_data['global_setting'] ) ) {
					$this->import_global_content_setting( $init_data['global_setting'] );
				}

				if ( ! empty( $init_data['users_data'] ) ) {
					$this->import_users_data( $init_data['users_data'] );
				}

				if ( isset( $init_data['categories'] ) ) {
					$this->import_categories( $init_data['categories'] );
					$this->create_categories();
					$this->set_categories( null );
				}

				if ( isset( $init_data['quiz_pro'] ) ) {
					if ( ! is_null( $index_to_import ) ) {
						$quiz_pro_total_count = $this->count_data( $init_data, 'quiz_pro' );

						$init_data['quiz_pro']['questions'] = array_slice( $init_data['quiz_pro']['questions'], $index_to_import, $batch_to_import );
					}
					$this->create_quiz_pro_data( $init_data['quiz_pro'] );
				}

				if ( isset( $init_data['posts'] ) ) {
					$this->import_posts( $init_data['posts'] );

					if ( ! is_null( $index_to_import ) ) {
						$current_posts    = $this->get_posts();
						$post_total_count = count( $current_posts );

						$new_posts = array_slice( $current_posts, $index_to_import, $batch_to_import );

						$this->set_posts( $new_posts );

					}

					$this->create_posts();
				}

				return [
					'success'        => true,
					'imported_count' => max( count( $init_data['categories'] ), count( $init_data['quiz_pro']['questions'] ), count( $this->get_posts() ) ),
					'total_count'    => max( $quiz_pro_total_count, $post_total_count ),
				];
			} catch ( \Exception $e ) {
				return [
					'success' => false,
				];
			}
		}

		/**
		 * Create Categories
		 *
		 * @return void
		 */
		private function create_categories() {
			foreach ( $this->get_categories() as $index => $category ) {
				$term = get_term_by( 'slug', $category['slug'], $category['term_taxonomy'] );

				if ( false !== $term ) {
					continue;
				}

				wp_insert_term(
					$category['name'],
					$category['term_taxonomy'],
					[
						'slug' => $category['slug'],
					]
				);
			}
		}


		/**
		 * Create Posts
		 *
		 * @return void
		 */
		private function create_posts() {
			$site_url = untrailingslashit( site_url() );

			foreach ( $this->get_posts() as $old_id => $post ) {
				$extra_metas   = [
					'ldie_old_post_id'  => $post['ld_old_post_id'],
					'ldie_to_be_import' => $post['ld_old_post_id'],
				];
				$ld_categories = isset( $post['ld_category'] ) ? $post['ld_category'] : [];
				$wp_post_meta  = isset( $post['wp_post_meta'] ) ? $post['wp_post_meta'] : [];

				unset( $post['ld_category'] );
				unset( $post['wp_post_meta'] );

				$post['guid']         = preg_replace( '@https?://[^/]+/@', $site_url . '/', $post['guid'] );
				$post['guid']         = str_replace( '{site_url}', $site_url, $post['guid'] );
				$post['post_content'] = str_replace( '{site_url}', $site_url, $post['post_content'] );

				$post['meta_input'] = $this->normalize_post_meta( $wp_post_meta, $extra_metas );

				$id = wp_insert_post( $post, false, false );
				if ( 0 === $id ) {
					continue;
				}

				$this->update_guid( $id, $post['guid'] );
				$this->create_post_categories( $id, $ld_categories );

				if ( 'sfwd-quiz' === $post['post_type'] ) {
					$quiz_pro_id = $this->create_quiz_pro( $id );
					update_post_meta( $id, 'ldie_new_quiz_pro_id', $quiz_pro_id );
				}

				$this->remove_unwanted_post_meta_from_db( 'ld_course_steps_dirty', $old_id );
				$this->remove_unwanted_post_meta_from_db( 'ld_course_steps_dirty', $id );

			}
		}

		private function update_guid( $post_id, $prev_guid ) {
			$site_url = untrailingslashit( site_url() ) . '/';
			$new_guid = preg_replace( '@https?://[^/]+/@', $site_url, $prev_guid );
			$new_guid = preg_replace( '@p=\d+@', "p={$post_id}", $new_guid );
			$new_guid = str_replace( '{site_url}', site_url(), $new_guid );
			// must update directly to the DB.
			wp_update_post(
				[
					'ID'   => $post_id,
					'guid' => $new_guid,
				]
			);
		}

		private function normalize_post_meta( $post_meta, $extra = [] ) {
			$normalized = [];
			foreach ( $post_meta as $meta ) {
				$row = (array) $meta;

				$data         = $row['meta_value'];
				$unserialized = @unserialize( $data ); // phpcs:ignore
				if ( false !== $unserialized ) {
					$normalized[ $row['meta_key'] ] = $unserialized;
					continue;
				}

				$json = json_decode( $row['meta_value'], true );
				if ( null !== $json && is_array( $json ) ) {
					$normalized[ $row['meta_key'] ] = wp_slash( $data );
					continue;
				}
				$normalized[ $row['meta_key'] ] = $data;
			}

			foreach ( $extra as $key => $value ) {
				$normalized[ $key ] = $value;
			}

			return $normalized;
		}

		/**
		 * Set Post Categories
		 *
		 * @param string $post_id .
		 * @param array  $raw_categories .
		 * @return void
		 */
		private function create_post_categories( $post_id, $raw_categories ) {
			$categories = [];

			foreach ( (array) $raw_categories as $index => $category ) {
				$term = get_term_by( 'slug', $category['ld_slug'], $category['ld_taxonomy'] );
				if ( false === $term ) {
					continue;
				}
				$categories[] = [
					'term_id'  => [ $term->term_id ],
					'taxonomy' => $term->taxonomy,
				];
			}

			foreach ( $categories as $index => $category ) {
				wp_set_post_terms( $post_id, $category['term_id'], $category['taxonomy'], true );
			}
		}

		/**
		 * Set the quiz pro first and then update the metadata
		 *
		 * @param integer $post_id .
		 * @return integer
		 */
		private function create_quiz_pro( $post_id ) {
			$post = get_post( $post_id );

			$quiz_mapper = new \WpProQuiz_Model_QuizMapper();
			$quiz_pro    = new \WpProQuiz_Model_Quiz();

			$quiz_pro->setName( $post->post_title );
			$quiz_pro->setText( 'AAZZAAZZ' );

			$quiz_pro = $quiz_mapper->save( $quiz_pro );

			$quiz_pro_id = $quiz_pro->getId();
			$quiz_pro_id = absint( $quiz_pro_id );

			return $quiz_pro_id;
		}

		/**
		 * Mapper for Old ids of quiz categories with new Ids
		 *
		 * @param array $quiz_pro_data .
		 * @return void
		 */
		private function create_quiz_pro_data( $quiz_pro_data ) {
			global $wpdb;

			$categories_map            = [];
			$questions_map             = [];
			$old_quiz_new_question_map = [];

			$raw_categories  = $quiz_pro_data['categories'];
			$category_mapper = new \WpProQuiz_Model_CategoryMapper();

			foreach ( $raw_categories as $category ) {
				$existing_category    = $category_mapper->fetchByName( $category['name'] );
				$existing_category_id = $existing_category->getCategoryId();
				if ( ! empty( $existing_category_id ) ) {
					$categories_map[ intval( $category['id'] ) ] = intval( $existing_category_id );
				} else {
					$quiz_pro_category = new \WpProQuiz_Model_Category();
					$quiz_pro_category->setCategoryName( $category['name'] );
					$category_mapper->save( $quiz_pro_category );
					$categories_map[ intval( $category['id'] ) ] = intval( $quiz_pro_category->getCategoryId() );
				}
			}

			$raw_questions = $quiz_pro_data['questions'];

			if ( empty( $raw_questions ) ) {
				return;
			}

			$table = $this->get_quiz_pro_database_table( 'question' );

			foreach ( $raw_questions as $row ) {
				$old_question_id    = $row['id'];
				$row['category_id'] = isset( $categories_map[ $row['category_id'] ] ) ? $categories_map[ $row['category_id'] ] : 0;
				$row['answer_data'] = stripslashes( $row['answer_data'] );

				unset( $row['id'] );
				$format = [];
				foreach ( $row as $val ) {
					$format[] = ( is_numeric( $val ) ) ? '%d' : '%s';
				}
				$result                                      = $wpdb->insert( $table, $row, $format ); // db call ok; no-cache ok.
				$lastid                                      = intval( $wpdb->insert_id );
				$questions_map[ strval( $old_question_id ) ] = $lastid;

				// Relating [ old_quiz_id ] => new question id.
				$old_quiz_new_question_map[ strval( $row['quiz_id'] ) ] = $lastid;
			}

			if ( ! empty( $old_quiz_new_question_map ) ) {
				$this->set_old_quiz_new_question_map( $old_quiz_new_question_map );
			}

			if ( ! empty( $questions_map ) ) {
				$this->set_quiz_questions_map( $questions_map );
			}

			if ( ! empty( $categories_map ) ) {
				$this->set_quiz_categories_map( $categories_map );
			}
		}

	}
}
