<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_Common_File' ) ) {
	/**
	 * Admin Import
	 */
	class Ld_Admin_Import_Common_File {


		/**
		 * Categories
		 *
		 * @var array
		 */
		protected $categories = [];

		/**
		 * Posts
		 *
		 * @var array
		 */
		protected $posts = [];

		/**
		 * Relation between new Post Ids and old (imported) Post Ids
		 *
		 * @var array
		 */
		protected $map_post_ids = [];

		/**
		 * Quiz Pro Questions map
		 *
		 * @var array
		 */
		protected $quiz_questions_map = [];

		/**
		 * Quiz Pro Old Quiz ids and New Questions Map
		 *
		 * @var array
		 */
		protected $old_quiz_new_question_map = [];

		/**
		 * Set categories
		 *
		 * @param array $categories .
		 * @return void
		 */
		public function set_categories( $categories ) {
			$this->categories = $categories;
		}

		/**
		 * Get Categories
		 *
		 * @return array
		 */
		public function get_categories() {
			return $this->categories;
		}

		/**
		 * Set Posts
		 *
		 * @param array $posts Posts.
		 *
		 * @return void
		 */
		public function set_posts( $posts ) {
			$this->posts = $posts;
		}

		/**
		 * Get Posts
		 *
		 * @return array
		 */
		public function get_posts() {
			return $this->posts;
		}

		/**
		 * Set Map Post IDs
		 *
		 * @param int $old_id New Post Id created.
		 * @param int $new_id Imported Post Id .
		 * @return void
		 */
		public function set_map_post_ids( $old_id, $new_id ) {
			$this->map_post_ids[ $old_id ] = $new_id;
		}

		/**
		 * Get map post Id
		 *
		 * @param int|null $search_id Post Id.
		 * @return int|string
		 */
		public function get_map_post_id( $search_id = null ) {
			if ( ! $search_id ) {
				return '';
			}

			return $this->get_new_post_id_by_old_post_id( $search_id );
		}

		/**
		 * Get new post ID from imported (old) post ID
		 *
		 * @param int $old_post_id Imported Post ID.
		 * @return int|string
		 */
		protected function get_new_post_id_by_old_post_id( $old_post_id ) {
			global $wpdb;

			$results = $wpdb->get_row( $wpdb->prepare( "SELECT post_id FROM {$wpdb->prefix}postmeta WHERE meta_key = 'ldie_old_post_id' AND meta_value = %s", $old_post_id ), ARRAY_A );
			return isset( $results['post_id'] ) ? $results['post_id'] : '';
		}

		/**
		 * Get all to be imported Post IDs from post meta
		 *
		 * @return array
		 */
		protected function get_all_post_ids_from_meta_key() {
			global $wpdb;

			$results = $wpdb->get_results( "SELECT post_id, meta_value FROM {$wpdb->prefix}postmeta WHERE meta_key = 'ldie_to_be_import' LIMIT 15", ARRAY_A );

			return $results;
		}

		/**
		 * Get all to be imported Post IDs from post meta for user
		 *
		 * @return array
		 */
		protected function get_all_post_ids_from_meta_key_for_user() {
			global $wpdb;

			$results = $wpdb->get_results( "SELECT post_id, meta_value FROM {$wpdb->prefix}postmeta WHERE meta_key = 'ldie_old_post_id'", ARRAY_A );

			return $results;
		}

		/**
		 * Get User IDs from imported (old) posts
		 *
		 * @return array
		 */
		protected function get_all_user_ids_from_imported_posts() {
			global $wpdb;

			$results = $wpdb->get_results( "SELECT p.ID as post_id, p.post_author FROM {$wpdb->prefix}posts p INNER JOIN {$wpdb->prefix}postmeta pm on p.ID = pm.post_id where pm.meta_key = 'ldie_old_post_id'", ARRAY_A );

			return $results;
		}

		/**
		 * Get old User IDs
		 *
		 * @return array
		 */
		protected function get_all_old_user_ids() {
			global $wpdb;

			$results = $wpdb->get_results( "SELECT user_id, meta_value FROM {$wpdb->prefix}usermeta where meta_key = 'ldie_old_user_id_to_be_import'  LIMIT 15", ARRAY_A );

			return $results;
		}

		/**
		 * Search for post ID
		 *
		 * @param string $id Current course progress Post ID.
		 * @param array  $array Array of new and old post IDs.
		 *
		 * @return string
		 */
		protected function search_for_id( $id, $array ) {
			if ( empty( $array ) || empty( $id ) ) {
				return '';
			}
			foreach ( $array as $key => $val ) {
				if ( (int) $val['meta_value'] === (int) $id ) {
					return $array[ $key ];
				}
			}
			return $id;
		}

		/**
		 * Run post meta mapper
		 *
		 * @return array
		 */
		public function run_post_metas_mapper() {
			$posts_from_meta = $this->get_all_post_ids_from_meta_key();

			foreach ( $posts_from_meta as $row ) {
				$old_id = $row['meta_value'];
				$new_id = intval( $row['post_id'] );

				$post_metas       = get_post_meta( $new_id );
				$post_metas_array = [];

				foreach ( $post_metas as $key => $value ) {
					$post_metas_array[] = [
						'meta_key'   => $key,
						'meta_value' => $value[0],
					];
				}

				$new_metas = $this->update_post_metadata( $new_id, $post_metas_array, $posts_from_meta );

				// Attachment post type.
				$post_type    = get_post_type( $new_id );
				$updated_post = [
					'ID'         => $new_id,
					'meta_input' => $new_metas,
				];

				if ( 'attachment' === $post_type ) {
					$parent_id                   = wp_get_post_parent_id( $new_id );
					$updated_post['post_parent'] = $this->get_map_post_id( $parent_id );
				}

				wp_update_post(
					$updated_post,
					false,
					false
				);

				delete_post_meta( $new_id, 'ldie_to_be_import' );
			}

			$is_last_batch = 0 === count( $posts_from_meta );

			return [
				'is_last_batch'     => $is_last_batch,
				'count_post_mapped' => count( $posts_from_meta ),
			];
		}

		/**
		 * Run user meta mapper
		 *
		 * @return array
		 */
		public function run_user_metas_mapper() {
			$imported_posts_id    = $this->get_all_user_ids_from_imported_posts();
			$old_and_new_user_ids = $this->get_all_old_user_ids();

			if ( ! empty( $old_and_new_user_ids ) && ! empty( $imported_posts_id ) ) {
				foreach ( $imported_posts_id as $post ) {
					foreach ( $old_and_new_user_ids as $row ) {
						$new_userid = (int) $row['user_id'];
						$old_userid = (int) $row['meta_value'];
						if ( $new_userid !== $old_userid ) {
							if ( $old_userid === $post['post_author'] ) {
								$arg = array(
									'ID'          => $post['post_id'],
									'post_author' => $new_userid,
								);
								wp_update_post( $arg );
							}
						}
					}
				}
			}

			$old_and_new_post_id = $this->get_all_post_ids_from_meta_key_for_user();

			if ( ! empty( $old_and_new_user_ids ) ) {
				foreach ( $old_and_new_user_ids as $row ) {
					$new_userid = (int) $row['user_id'];
					$old_userid = (int) $row['meta_value'];

					$course_ids = learndash_get_user_courses_from_meta( $new_userid );
					if ( ! empty( $course_ids ) ) {
						foreach ( $course_ids as $course_id ) {
							$user_course_access_time = get_user_meta( $new_userid, 'course_' . $course_id . '_access_from', true );
							$old_course_ids          = $this->search_for_id( $course_id, $old_and_new_post_id );

							if ( (int) $old_course_ids['meta_value'] === (int) $course_id ) {
								delete_user_meta( $new_userid, 'course_' . $course_id . '_access_from' );
								$new_course_id = $old_course_ids['post_id'];
								update_user_meta( $new_userid, 'course_' . $new_course_id . '_access_from', $user_course_access_time );
							}
						}
					}
				}
			}

			if ( ! empty( $old_and_new_user_ids ) ) {
				foreach ( $old_and_new_user_ids as $user ) {
					$new_user_id = $user['user_id'];

					$course_progress = get_user_meta( $new_user_id, '_sfwd-course_progress', true );

					if ( empty( $course_progress ) ) {
						delete_user_meta( $new_user_id, 'ldie_old_user_id_to_be_import' );
						continue;
					}

					$course_progress = maybe_unserialize( $course_progress );
					$courses_keys    = array_keys( $course_progress );

					$user_course_progression = array();

					foreach ( $courses_keys as $courses_key ) {
						if ( empty( $courses_key ) ) {
							continue;
						}

						$old_course_ids      = $this->search_for_id( $courses_key, $old_and_new_post_id );
						$course_progress_tmp = array();

						if ( ! empty( $old_course_ids ) ) {
							$new_course_id = ! empty( $old_course_ids['post_id'] ) ? $old_course_ids['post_id'] : $old_course_ids;

							$lessons_keys = '';
							if ( ! empty( $course_progress[ $courses_key ]['lessons'] ) ) {
								$lessons_keys = array_keys( $course_progress[ $courses_key ]['lessons'] );
							}

							if ( empty( $lessons_keys ) ) {
								continue;
							}

							foreach ( $lessons_keys as $lessons_key ) {

								if ( empty( $lessons_key ) ) {
									continue;
								}
								$old_lesson_ids = $this->search_for_id( $lessons_key, $old_and_new_post_id );

								if ( ! empty( $old_lesson_ids ) ) {
									$new_lesson_id = ! empty( $old_lesson_ids['post_id'] ) ? $old_lesson_ids['post_id'] : $old_lesson_ids;
									$lesson_value  = $course_progress[ $courses_key ]['lessons'][ $lessons_key ];

									$course_progress_tmp['lessons'][ $new_lesson_id ] = $lesson_value;

									$topics_keys = '';
									if ( ! empty( $course_progress[ $courses_key ]['topics'][ $lessons_key ] ) ) {
										$topics_keys = array_keys( $course_progress[ $courses_key ]['topics'][ $lessons_key ] );
									}

									if ( empty( $topics_keys ) ) {
										continue;
									}

									foreach ( $topics_keys as $topics_key ) {

										if ( empty( $topics_key ) ) {
											continue;
										}

										$old_topic_ids = $this->search_for_id( $topics_key, $old_and_new_post_id );

										if ( ! empty( $old_topic_ids ) ) {
											$new_topic_id = ! empty( $old_topic_ids['post_id'] ) ? $old_topic_ids['post_id'] : $old_topic_ids;
											$topic_value  = $course_progress[ $courses_key ]['topics'][ $lessons_key ][ $topics_key ];
											$course_progress_tmp['topics'][ $new_lesson_id ][ $new_topic_id ] = $topic_value;
										}
									}
								}
							}
							$user_course_progression[ $new_course_id ] = $course_progress_tmp;
						}
					}
					update_user_meta( $new_user_id, '_sfwd-course_progress', $user_course_progression );

					delete_user_meta( $new_user_id, 'ldie_old_user_id_to_be_import' );
				}
			}

			$is_last_batch = 0 === count( $old_and_new_user_ids );

			if ( $is_last_batch ) {
				$this->remove_unwanted_post_meta_from_db( 'ldie_is_updated' );
				$this->remove_unwanted_post_meta_from_db( 'ldie_old_post_id' );
				$this->remove_unwanted_post_meta_from_db( 'ld_course_steps_dirty' );
				$this->remove_unwanted_post_meta_from_db( 'ld_quiz_questions_dirty' );
				$this->remove_unwanted_user_meta_from_db( 'ldie_old_user_id' );
				$this->remove_unwanted_user_meta_from_db( 'ldie_old_user_id_to_be_import' );
			}

			return [
				'is_last_batch'     => $is_last_batch,
				'count_user_mapped' => count( $old_and_new_user_ids ),
			];

		}

		/**
		 * Clean up after cancelled import process.
		 *
		 * @return void
		 */
		public function clean_canceled_import_process() {
			$this->remove_unwanted_post_meta_from_db( 'ldie_to_be_import' );
			$this->remove_unwanted_post_meta_from_db( 'ldie_old_post_id' );
			$this->remove_unwanted_post_meta_from_db( 'ldie_is_updated' );
			$this->remove_unwanted_user_meta_from_db( 'ldie_old_user_id' );
		}

		/**
		 * Remove post meta from database.
		 *
		 * @param string $key     Meta key.
		 * @param int    $post_id Post ID.
		 *
		 * @return void
		 */
		protected function remove_unwanted_post_meta_from_db( $key, $post_id = null ) {
			global $wpdb;
			if ( $post_id ) {
				// $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}postmeta WHERE post_id = %d AND meta_key = %s", $post_id, $key ) );
				$wpdb->delete(
					"{$wpdb->prefix}postmeta",
					[
						'post_id'  => $post_id,
						'meta_key' => $key,
					],
					[
						'%d',
						'%s',
					]
				);
			} else {
				// $wpdb->query( $wpdb->prepare( "DELETE FROM {$wpdb->prefix}postmeta WHERE meta_key = %s", $key ) );
				$wpdb->delete(
					"{$wpdb->prefix}postmeta",
					[
						'meta_key' => $key,
					],
					[
						'%s',
					]
				);
			}
		}

		/**
		 * Remove user meta from database
		 *
		 * @param string $key     Meta key.
		 * @param int    $user_id User ID.
		 *
		 * @return void
		 */
		protected function remove_unwanted_user_meta_from_db( $key, $user_id = null ) {
			global $wpdb;
			if ( $user_id ) {
				$wpdb->delete(
					"{$wpdb->prefix}usermeta",
					[
						'user_id'  => $user_id,
						'meta_key' => $key,
					],
					[
						'%d',
						'%s',
					]
				);
			} else {
				$wpdb->delete(
					"{$wpdb->prefix}usermeta",
					[
						'meta_key' => $key,
					],
					[
						'%s',
					]
				);
			}
		}

		/**
		 * Get WP ProQuiz Database table name
		 *
		 * Newer versions of LearnDash are using a different table name for quiz data.
		 *
		 * @param string $type Type of WP ProQuiz table to look for. Default: question.
		 *
		 * @return string
		 */
		protected function get_quiz_pro_database_table( $type ) {
			global $wpdb;

			$table_name = "{$wpdb->prefix}learndash_pro_quiz_{$type}";

			if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) === $table_name ) {
				return $table_name;
			}

			return "{$wpdb->prefix}wp_pro_quiz_{$type}";
		}

		/**
		 * Set Post metadata
		 *
		 * @param int   $post_id         Post ID.
		 * @param array $post_meta       Post meta.
		 * @param array $posts_from_meta Posts from meta.
		 *
		 * @return array
		 */
		protected function update_post_metadata( $post_id, $post_meta, $posts_from_meta ) {
			$new_metas       = [];
			$quiz_pro_id     = $this->get_new_quiz_pro_id( $post_id );
			$already_updated = get_post_meta( $post_id, 'ldie_is_updated', false );

			foreach ( $post_meta as $meta ) {
				$row = (array) $meta;

				if ( preg_match( '/ld_course_(\d+)/', $row['meta_key'], $matches ) ) {
					$new_post_id_key = $this->get_map_post_id( $matches[1] );
					$new_post_id     = $this->get_map_post_id( $row['meta_value'] );
					$new_key         = "ld_course_{$new_post_id_key}";

					if ( $new_post_id_key && $new_post_id ) {
						$new_metas[ $new_key ] = $new_post_id;
						delete_post_meta( $post_id, $row['meta_key'] );
					}

					continue;
				}

				switch ( $row['meta_key'] ) {

					case 'course_id':
						if ( ! $already_updated ) {
							$get_new_parent                = $this->get_map_post_id( $row['meta_value'] );
							$new_metas[ $row['meta_key'] ] = $get_new_parent;
						}
						break;

					case 'lesson_id':
						if ( ! $already_updated ) {
							$get_new_parent                = $this->get_map_post_id( $row['meta_value'] );
							$new_metas[ $row['meta_key'] ] = $get_new_parent;
						}
						break;

					case '_sfwd-lessons':
					case '_sfwd-lessons_course':
						$get_data                        = unserialize($row['meta_value']); // phpcs:ignore
						if ( isset( $get_data['sfwd-lessons_course'] ) ) {
							$get_new_parent                  = $this->get_map_post_id( $get_data['sfwd-lessons_course'] );
							$get_data['sfwd-lessons_course'] = $get_new_parent;
						}

						if ( isset( $get_data['sfwd-lessons_lesson_materials'] ) ) {
							$get_data['sfwd-lessons_lesson_materials'] = str_replace( '{site_url}', site_url(), $get_data['sfwd-lessons_lesson_materials'] );
						}

						$new_metas[ $row['meta_key'] ] = $get_data;
						break;

					case '_sfwd-topic':
						$get_data                      = unserialize($row['meta_value']); // phpcs:ignore
						if ( ! $already_updated ) {
							if ( isset( $get_data['sfwd-topic_course'] ) ) {
								$get_data['sfwd-topic_course'] = $this->get_map_post_id( $get_data['sfwd-topic_course'] );
							}
							if ( isset( $get_data['sfwd-topic_lesson'] ) ) {
								$get_data['sfwd-topic_lesson'] = $this->get_map_post_id( $get_data['sfwd-topic_lesson'] );
							}
						}

						if ( isset( $get_data['sfwd-topic_materials'] ) ) {
							$get_data['sfwd-topic_materials'] = str_replace( '{site_url}', site_url(), $get_data['sfwd-topic_materials'] );
						}

						$new_metas[ $row['meta_key'] ] = $get_data;
						break;

					case '_sfwd-quiz':
						$get_data                     = unserialize($row['meta_value']); // phpcs:ignore
						if ( ! $already_updated ) {
							if ( isset( $get_data['sfwd-quiz_course'] ) ) {
								$get_new_parent               = $this->get_map_post_id( $get_data['sfwd-quiz_course'] );
								$get_data['sfwd-quiz_course'] = $get_new_parent;
							}

							if ( isset( $get_data['sfwd-quiz_lesson'] ) ) {
								$get_lesson_parent            = $this->get_map_post_id( $get_data['sfwd-quiz_lesson'] );
								$get_data['sfwd-quiz_lesson'] = $get_lesson_parent;
							}
						}

						if ( isset( $get_data['sfwd-quiz_certificate'] ) ) {
							$get_cert_id                       = $this->get_map_post_id( $get_data['sfwd-quiz_certificate'] );
							$get_data['sfwd-quiz_certificate'] = $get_cert_id;
						}

						if ( isset( $get_data['sfwd-quiz_materials'] ) ) {
							$get_data['sfwd-quiz_materials'] = str_replace( '{site_url}', site_url(), $get_data['sfwd-quiz_materials'] );
						}

						$get_data['sfwd-quiz_quiz_pro'] = $quiz_pro_id;

						$new_metas[ $row['meta_key'] ] = $get_data;
						break;

					case '_ld_certificate':
						$new_metas[ $row['meta_key'] ] = $this->get_map_post_id( $row['meta_value'] );
						break;

					case 'quiz_pro_id':
						$get_prev_pro_id = intval( $row['meta_value'] );
						delete_post_meta( $post_id, 'quiz_pro_primary_' . $get_prev_pro_id );
						delete_post_meta( $post_id, 'quiz_pro_id_' . $get_prev_pro_id );

						// re assign.
						$new_metas[ 'quiz_pro_primary_' . $quiz_pro_id ] = $quiz_pro_id;
						$new_metas[ 'quiz_pro_id_' . $quiz_pro_id ]      = $quiz_pro_id;
						$new_metas['quiz_pro_id']                        = $quiz_pro_id;

						$this->update_quiz_pro( $get_prev_pro_id, $quiz_pro_id );
						break;

					case 'quiz_id':
						$get_new_parent                = $this->get_map_post_id( $row['meta_value'] );
						$new_metas[ $row['meta_key'] ] = $get_new_parent;
						break;

					case '_sfwd-question':
						$get_data                       = unserialize($row['meta_value']); // phpcs:ignore
						if ( isset( $get_data['sfwd-question_quiz'] ) ) {
							$get_new_parent                 = $this->get_map_post_id( $get_data['sfwd-question_quiz'] );
							$get_data['sfwd-question_quiz'] = $get_new_parent;
						}
						$new_metas[ $row['meta_key'] ] = $get_data;
						break;

					case 'question_pro_category':
						// Searching for the old category, to get the new one.
						$categories                    = $this->get_quiz_categories_map();
						$new_metas[ $row['meta_key'] ] = isset( $categories[ strval( $row['meta_value'] ) ] ) ? $categories[ strval( $row['meta_value'] ) ] : '';
						break;
					case 'question_pro_id':
						// Searching for the old question pro id, to get the new one.
						$questions_map                 = $this->get_quiz_questions_map();
						$new_metas[ $row['meta_key'] ] = $questions_map[ strval( $row['meta_value'] ) ];
						break;

					case 'ld_quiz_questions':
						$get_data = unserialize($row['meta_value']); // phpcs:ignore
						$new_data = [];

						$questions_map = $this->get_quiz_questions_map();

						foreach ( $get_data as $old_post_id => $old_question_id ) {
							$new_question_id          = isset( $questions_map[ strval( $old_question_id ) ] ) ? $questions_map[ strval( $old_question_id ) ] : '';
							$new_post_id              = $this->get_map_post_id( $old_post_id );
							$new_data[ $new_post_id ] = $new_question_id;
						}
						$new_metas[ $row['meta_key'] ] = $new_data;
						break;

					case 'quiz_pro_primary_' . intval( $row['meta_value'] ):
					case 'quiz_pro_id_' . intval( $row['meta_value'] ):
						break;

					case '_sfwd-courses':
						$get_data = unserialize( $row['meta_value'] ); // phpcs:ignore

						if ( isset( $get_data['groups_certificate'] ) ) {
							$get_data['groups_certificate'] = $this->get_map_post_id( $get_data['groups_certificate'] );
						}

						if ( isset( $get_data['sfwd-courses_course_materials'] ) ) {
							$get_data['sfwd-courses_course_materials'] = str_replace( '{site_url}', site_url(), $get_data['sfwd-courses_course_materials'] );
						}

						if ( isset( $get_data['sfwd-courses_certificate'] ) ) {
							$get_data['sfwd-courses_certificate'] = $this->get_map_post_id( $get_data['sfwd-courses_certificate'] );
						}

						$new_metas[ $row['meta_key'] ] = $get_data;
						break;

					case '_groups':
						$get_data                       = unserialize( $row['meta_value'] ); // phpcs:ignore
						$get_data['groups_certificate'] = $this->get_map_post_id( $get_data['groups_certificate'] );
						$new_metas[ $row['meta_key'] ]  = $get_data;
						break;
					case 'ld_course_steps':
						$get_data              = unserialize( $row['meta_value'] ); // phpcs:ignore
						$get_data['course_id'] = $post_id;
						if ( isset( $get_data['steps']['h'] ) ) {
							$get_data['steps']['h'] = $this->update_course_steps( $get_data['steps']['h'], $post_id );
						} elseif ( isset( $get_data['h'] ) ) {
							$get_data['h'] = $this->update_course_steps( $get_data['h'], $post_id );

						}
						$new_metas[ $row['meta_key'] ] = $get_data;
						delete_post_meta( $post_id, 'ld_course_steps_dirty' );
						break;
					case '_thumbnail_id':
						$new_metas[ $row['meta_key'] ] = $this->get_map_post_id( $row['meta_value'] );
						break;

					case 'learndash_group_enrolled_' === substr( $row['meta_key'], 0, strlen( 'learndash_group_enrolled_' ) ):
						$old_group_id = preg_replace( '/[^0-9]/', '', $row['meta_key'] );
						if ( is_array( $posts_from_meta ) ) {
							foreach ( $posts_from_meta as $key ) {
								if ( $key['meta_value'] === $old_group_id ) {
									delete_post_meta( $post_id, $row['meta_key'] );
									$row['meta_key']               = str_replace( $old_group_id, $key['post_id'], $row['meta_key'] );
									$new_metas[ $row['meta_key'] ] = $row['meta_value'];
									break;
								}
							}
						}
						break;

					default:
						$data         = $row['meta_value'];
						$unserialized = @unserialize( $data ); // phpcs:ignore
						if ( false !== $unserialized ) {
							$new_metas[ $row['meta_key'] ] = $unserialized;
							break;
						}

						$json = json_decode( $row['meta_value'], true );

						if ( null !== $json && is_array( $json ) ) {
							$new_metas[ $row['meta_key'] ] = wp_slash( $data );
							break;
						}

						$new_metas[ $row['meta_key'] ] = wp_slash( $data );
						break;
				}
			}

			return $new_metas;
		}

		/**
		 * Update course progress
		 *
		 * @param array  $old_steps       Array of old course steps.
		 * @param int    $course_id       Course ID.
		 * @param string $type            Post type.
		 * @param int    $lesson_id       Lesson ID.
		 * @param int    $topic_id        Topic ID.
		 * @param int    $current_post_id Current Post ID.
		 *
		 * @return array
		 */
		protected function update_course_steps( $old_steps, $course_id, $type = null, $lesson_id = null, $topic_id = null, $current_post_id = null ) {
			$new_steps = [];

			foreach ( $old_steps as $mixed => $array_steps ) {
				if ( preg_match( '/sfwd-(\w+)/', $mixed, $matches ) ) {
					$new_key = $mixed;
					$type    = $matches[1];
				} else {
					$new_key         = intval( $this->get_map_post_id( $mixed ) );
					$current_post_id = $new_key;

					if ( 'lessons' === $type ) {
						$lesson_id = $new_key;
						$topic_id  = null;
					}
					if ( 'topic' === $type ) {
						$topic_id = $new_key;
					}

					if ( ! empty( $type ) && ! empty( $current_post_id ) && ! empty( $lesson_id ) ) {
						$this->add_metas_for_course_steps( $current_post_id, $course_id, $type, $lesson_id, $topic_id );
					}
				}

				$new_steps[ $new_key ] = count( $array_steps ) > 0 ? $this->update_course_steps( $array_steps, $course_id, $type, $lesson_id, $topic_id, $current_post_id ) : [];
			}

			return $new_steps;
		}

		/**
		 * Add meta data for course steps
		 *
		 * @param int    $current_post_id Current Post ID.
		 * @param int    $course_id       Course ID.
		 * @param string $type            Post type.
		 * @param int    $lesson_id       Lesson ID.
		 * @param int    $topic_id        Topic ID.
		 *
		 * @return void
		 */
		protected function add_metas_for_course_steps( $current_post_id, $course_id, $type, $lesson_id, $topic_id ) {

			update_post_meta( $current_post_id, 'ldie_is_updated', true );

			if ( 'lessons' === $type ) {
				update_post_meta( $current_post_id, 'course_id', $course_id );
				return;
			}

			update_post_meta( $current_post_id, 'course_id', $course_id );

			if ( 'topic' === $type ) {
				update_post_meta( $current_post_id, 'lesson_id', $lesson_id );
				$sfwd_topic = get_post_meta( $current_post_id, '_sfwd-topic', true );

				$sfwd_topic['sfwd-topic_course'] = $course_id;
				$sfwd_topic['sfwd-topic_lesson'] = $lesson_id;
				update_post_meta( $current_post_id, '_sfwd-topic', $sfwd_topic );
				return;
			}

			if ( 'quiz' === $type ) {
				$new_id = null === $topic_id ? $lesson_id : $topic_id;

				update_post_meta( $current_post_id, 'lesson_id', $new_id );
				$sfwd_quiz = get_post_meta( $current_post_id, '_sfwd-quiz', true );

				$sfwd_quiz['sfwd-quiz_course'] = $course_id;
				$sfwd_quiz['sfwd-quiz_lesson'] = $new_id;
				update_post_meta( $current_post_id, '_sfwd-quiz', $sfwd_quiz );
				return;
			}
		}

		/**
		 * Set questions to database and update the quiz_id later
		 *
		 * @param int $old_quiz_id Old Quiz ID.
		 * @param int $quiz_id     Quiz ID.
		 *
		 * @return void
		 */
		protected function update_quiz_pro( $old_quiz_id = 0, $quiz_id ) {
			global $wpdb;

			$table = $this->get_quiz_pro_database_table( 'question' );

			$old_quiz_new_question_map = $this->get_old_quiz_new_question_map();

			if ( isset( $old_quiz_new_question_map[ strval( $old_quiz_id ) ] ) ) {
				$wpdb->update(
					$table,
					[
						'quiz_id' => intval( $quiz_id ),
					],
					[
						'id' => $old_quiz_new_question_map[ strval( $old_quiz_id ) ],
					],
					[
						'%d',
					],
					[
						'%d',
					]
				); // db call ok; no-cache ok.
			}
		}

		/**
		 * Set quiz categories map.
		 *
		 * @param array $quiz_categories_map Quiz categories map.
		 *
		 * @return void
		 */
		public function set_quiz_categories_map( $quiz_categories_map ) {
			$current = $this->get_quiz_categories_map();

			$new_data = array_merge( $current, $quiz_categories_map );
			update_option( 'ldie_quiz_categories_map', $new_data );
		}

		/**
		 * Get quiz categories map.
		 *
		 * @return array
		 */
		public function get_quiz_categories_map() {
			return get_option( 'ldie_quiz_categories_map', [] );
		}

		/**
		 * Set quiz questions map.
		 *
		 * @param array $quiz_questions_map Quiz questions map.
		 *
		 * @return void
		 */
		public function set_quiz_questions_map( $quiz_questions_map ) {
			$current = $this->get_quiz_questions_map();

			foreach ( $quiz_questions_map as $index => $value ) {
				$current[ $index ] = $value;
			}

			update_option( 'ldie_quiz_questions_map', $current );
		}

		/**
		 * Get quiz questions map.
		 *
		 * @return array
		 */
		public function get_quiz_questions_map() {

			if ( empty( $this->quiz_questions_map ) ) {
				$this->quiz_questions_map = get_option( 'ldie_quiz_questions_map', [] );
			}

			return $this->quiz_questions_map;
		}

		/**
		 * Set old quiz to new questions map.
		 *
		 * @param array $old_quiz_new_question_map Old quiz to new questions map.
		 *
		 * @return void
		 */
		public function set_old_quiz_new_question_map( $old_quiz_new_question_map ) {
			$current = $this->get_old_quiz_new_question_map();

			foreach ( $old_quiz_new_question_map as $index => $value ) {
				$current[ $index ] = $value;
			}
			update_option( 'ldie_old_quiz_new_question_map', $current );
		}

		/**
		 * Get old quiz to new questions map.
		 *
		 * @return array
		 */
		public function get_old_quiz_new_question_map() {

			if ( empty( $this->old_quiz_new_question_map ) ) {
				$this->old_quiz_new_question_map = get_option( 'ldie_old_quiz_new_question_map', [] );
			}

			return $this->old_quiz_new_question_map;
		}

		/**
		 * Get the the new quiz pro attached on a post meta.
		 *
		 * @param int $post_id Post ID.
		 *
		 * @return string
		 */
		public function get_new_quiz_pro_id( $post_id ) {
			return get_post_meta( $post_id, 'ldie_new_quiz_pro_id', true );
		}
	}
}
