<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_File' ) ) {
	/**
	 * Admin Import
	 */
	class Ld_Admin_Import_File {

		protected $import_process;
		/**
		 * Construct
		 */
		public function __construct() {
			// is this necessary?
			$this->text        = Ld_Text::text( 'notices' );
			$this->file_errors = Ld_Text::text( 'file' );

			add_action( 'wp_ajax_ldie_upload_import_file', [ $this, 'upload_file' ] );
			add_action( 'wp_ajax_ldie_import_status', [ $this, 'get_process_status' ] );
			add_action( 'wp_ajax_ldie_import_cancel_queue', [ $this, 'cancel_queue' ] );
			$this->import_process = new Ld_Admin_Import_Process_Actions();

			$this->import_process->add_actions();
		}

		public function cancel_queue() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_import_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$this->import_process->run_import_cleaner();
				$importer = new Ld_Admin_Import_Common_File();
				$importer->clean_canceled_import_process();

				echo wp_json_encode(
					[
						'status' => 'success',
					]
				);
			} catch ( \Exception $error ) {
				echo wp_json_encode(
					[
						'status' => 'error',
					]
				);
			}
			exit();
		}

		public function get_process_status() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_import_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$current_process = isset( $_POST['current_process'] ) ? sanitize_text_field( wp_unslash( $_POST['current_process'] ) ) : '';

				$data = get_option( Ld_Admin_Import_Process_Actions::PROCESS_OPTION, [] );

				$message = '';
				$status  = 'success';

				$error_log = $this->get_failed_task_error();

				if ( ! is_null( $error_log ) ) {
					$data['completed'] = true;
					update_option( Ld_Admin_Import_Process_Actions::PROCESS_OPTION, $data );
				}

				if ( isset( $data['completed'] ) && $data['completed'] ) {
					$message = 'Data is successfully imported';
					$this->import_process->run_import_cleaner();

					if ( ! is_null( $error_log ) ) {
						$status  = 'error';
						$message = $error_log;

						$data['process'][ $current_process ]['status'] = 'error';
					}
				}

				echo wp_json_encode(
					[
						'status'  => $status,
						'data'    => $data,
						'process' => isset( $data['process'] ) ? $data['process'][ $current_process ] : [],
						'message' => $message,
					]
				);
			} catch ( \Exception $error ) {
				$this->import_process->run_import_cleaner();
				echo wp_json_encode(
					[
						'status'  => 'error',
						'message' => $error->getMessage(),
					]
				);
			}
			exit();
		}

		private function get_action_status( $action_id ) {
			return \ActionScheduler::store()->get_status( $action_id );
		}

		/**
		 * Do import
		 *
		 * @throws \Exception When the file is invalid or when the imported data failed.
		 * @return void
		 */
		public function upload_file() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_import_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$type           = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : 'zip';
				$is_slow_run    = isset( $_POST['is_slow_run'] ) ? sanitize_text_field( wp_unslash( $_POST['is_slow_run'] ) ) === '1' : false;
				$slow_run_batch = isset( $_POST['slow_run_batch'] ) ? (int) sanitize_text_field( wp_unslash( $_POST['slow_run_batch'] ) ) : 1;
				$files_uploaded = null;

				if ( 'zip' === $type ) {

					if ( isset( $_FILES['file']['error'] ) && 0 < $_FILES['file']['error'] ) {
						throw new \Exception( $this->file_errors['error'][ sanitize_text_field( wp_unslash( $_FILES['file']['error'] ) ) ] );
					}

					if ( ! isset( $_FILES['file'] ) || ( isset( $_FILES['file']['name'] ) && ! sanitize_text_field( wp_unslash( $_FILES['file']['name'] ) ) ) ) {
						throw new \Exception( esc_html__( 'No file is found', 'ea_import_export' ) );
					}

					$tmp_name  = isset( $_FILES['file']['tmp_name'] ) ? $_FILES['file']['tmp_name'] : '';
					$data_type = isset( $_FILES['file']['type'] ) ? sanitize_text_field( wp_unslash( $_FILES['file']['type'] ) ) : '';
					$file_name = sanitize_text_field( wp_unslash( $_FILES['file']['name'] ) );

					if ( ! $tmp_name && ! $data_type ) {
						throw new \Exception( esc_html__( 'No file is submited', 'ea_import_export' ) );
					}

					if ( 'application/zip' === $data_type || 'application/x-zip-compressed' === $data_type || strpos( $data_type, 'zip' ) !== false ) {
						$files_uploaded = Ld_Admin_File_System::decompress_zip_file( $tmp_name, $file_name );
					} else {
						$files_uploaded = Ld_Admin_File_System::upload_file( $tmp_name, $file_name, $data_type );
					}
				} elseif ( 'dir' === $type ) {
					$directory      = isset( $_POST['directory_path'] ) ? sanitize_text_field( wp_unslash( $_POST['directory_path'] ) ) : null;
					$files_uploaded = Ld_Admin_File_System::scan_import_directory( $directory );

				} else {
					throw new \Exception( esc_html__( 'Option is not valid', 'ea_import_export' ) );
				}

				if ( empty( $files_uploaded ) ) {
					throw new \Exception( esc_html__( 'Invalid file to import', 'ea_import_export' ) );
				}

				$date = new \DateTime( 'now', new \DateTimeZone( 'UTC' ) );

				$queue_started_at = $date->getTimestamp();
				as_schedule_recurring_action( $queue_started_at, 3, Ld_Admin_Import_Process_Actions::ANALYZER_ACTION, [], Ld_Admin_Import_Process_Actions::ACTION_GROUP );
				$this->create_option_for_import_data(
					$files_uploaded,
					$queue_started_at,
					[
						'is_slow_run' => [
							'enabled' => $is_slow_run,
							'context' => [
								'batch_to_run'   => $slow_run_batch,
								'total_count'    => 0,
								'imported_count' => 0,
							],
						],
					]
				);

				echo wp_json_encode(
					[
						'status' => 'success',
						'data'   => $files_uploaded,
					]
				);
			} catch ( \Exception $e ) {
				wp_send_json_error(
					[
						'status'  => 'error',
						'message' => $e->getMessage(),
					],
					500
				);
			}
			exit();
		}

		private function create_option_for_import_data( $uploaded, $queue_started_at, $settings ) {
			update_option(
				'ldie_import_data',
				[
					'completed'        => false,
					'dir'              => $uploaded['dir'],
					'files'            => $uploaded['files'],
					'queue_started_at' => $queue_started_at,
					'settings'         => $settings,
				]
			);
		}

		private function get_failed_task_error() {
			$option       = get_option( 'ldie_import_data' );
			$failed_tasks = [];

			if ( isset( $option['queue_started_at'] ) ) {
				$timezone   = new \DateTimeZone( 'UTC' );
				$started_at = new \DateTime();
				$started_at->setTimezone( $timezone );
				$started_at->setTimestamp( $option['queue_started_at'] );

				$failed_tasks = as_get_scheduled_actions(
					[
						'group'        => Ld_Admin_Import_Process_Actions::ACTION_GROUP,
						'status'       => \ActionScheduler_Store::STATUS_FAILED,
						'order'        => 'DESC',
						'date'         => $started_at,
						'date_compare' => '>=',
					],
					'ids'
				);

				foreach ( $failed_tasks as $failed_id ) {
					$logs = \ActionScheduler::logger()->get_logs( $failed_id );

					foreach ( $logs as $log ) {
						$message = $log->get_message();

						if ( preg_match( '@(timed out)|(unexpected shutdown|corrupted|invalid)@', $message ) ) {
							return $message;
						}
					}
				}
			}

			return null;
		}

	}
}
