<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_JSON_File' ) ) {

	/**
	 * Admin Import
	 */
	class Ld_Admin_Import_JSON_File extends Ld_Admin_Import_Abstract_File {

		/**
		 * Abstract
		 *
		 * @param string $filepath .
		 * @return mixed
		 * @throws \Exception .
		 */
		protected function prepare_file_data( $filepath ) {
			$data = json_decode(file_get_contents($filepath), true); // phpcs:ignore

			if ( ! $this->is_valid_data( $data ) ) {
				throw new \Exception( esc_html__( 'Corrupted or Invalid data', 'ea_import_export' ) );
			}

			return $data;
		}

		/**
		 * Abstract
		 *
		 * @param array $data .
		 * @return boolean
		 */
		protected function is_valid_data( $data ) {
			return md5( 'ldie' ) === $data['keyid'];
		}


		protected function unserialize_questions( $raw_questions ) {

			$questions = [];

			foreach ( $raw_questions as $question ) {
				$questions[] = unserialize( $question );
			}

			return $questions;

		}

		/**
		 * Abstract function to initialize the structure
		 *
		 * @param array $data .
		 * @return array
		 */
		protected function init( $data ) {
			$init_data = [];
			$init_data['global_setting'] = isset( $data['ld_global_setting'] ) ? $data['ld_global_setting'] : [];
			$init_data['users_data']     = isset( $data['wp_users_data'] ) ? $data['wp_users_data'] : [];
			$init_data['categories']     = isset( $data['wp_categories'] ) ? $data['wp_categories'] : [];
			$init_data['quiz_pro']       = [
				'questions'  => $this->unserialize_questions( isset( $data['quiz_pro']['questions'] ) ? $data['quiz_pro']['questions'] : [] ),
				'categories' => isset( $data['quiz_pro']['categories'] ) ? $data['quiz_pro']['categories'] : [],
			];
			$init_data['posts']          = isset( $data['wp_data'] ) ? $data['wp_data'] : [];

			return $init_data;
		}

		/**
		 * Import the Categories
		 *
		 * @param array $init_categories .
		 * @return void
		 */
		protected function import_categories( $init_categories ) {
			$categories = [];

			foreach ( $init_categories as $category ) {
				$categories[] = [
					'term_id'         => $category['wp_term_id'],
					'term_taxonomy'   => $category['wp_term_taxonomy'],
					'slug'            => $category['wp_category_nicename'],
					'category_parent' => $category['wp_category_parent'],
					'name'            => $category['wp_category_name'],
				];
			}

			$this->set_categories( $categories );
		}

		/**
		 * Create the Posts
		 *
		 * @param object $raw_posts .
		 * @return void
		 */
		protected function import_posts( $raw_posts ) {
			$posts = [];

			$postmeta_callback = function( $meta ) {
				$as_array = (array) $meta;
				return [
					'meta_key'   => (string) $as_array['wp_meta_key'], // phpcs:ignore
					'meta_value' => (string) $as_array['wp_meta_value'], // phpcs:ignore
				];
			};

			foreach ( $raw_posts as $index => $raw_post ) {
				$post = [
					'post_author'           => $raw_post['wp_author_id'],
					'post_date'             => $raw_post['wp_post_date'],
					'post_date_gmt'         => $raw_post['wp_post_date_gmt'],
					'post_content'          => $raw_post['wp_post_content'],
					'post_content_filtered' => wp_strip_all_tags( $raw_post['wp_post_content'] ),
					'post_title'            => $raw_post['wp_post_title'],
					'post_excerpt'          => $raw_post['wp_post_excerpt'],
					'post_status'           => $raw_post['wp_post_status'],
					'post_type'             => $raw_post['wp_post_type'],
					'comment_status'        => $raw_post['wp_comment_status'],
					'ping_status'           => $raw_post['wp_ping_status'],
					'post_password'         => $raw_post['wp_post_password'],
					'post_name'             => $raw_post['wp_post_name'],
					'to_ping'               => $raw_post['wp_to_ping'],
					'pinged'                => $raw_post['wp_pinged'],
					'post_modified'         => $raw_post['wp_post_modified'],
					'post_modified_gmt'     => $raw_post['wp_post_modified_gmt'],
					'post_parent'           => $raw_post['wp_post_parent'],
					'menu_order'            => $raw_post['wp_menu_order'],
					'post_mime_type'        => $raw_post['wp_post_mime_type'],
					'guid'                  => $raw_post['guid']['value'],
					'ld_old_post_id'        => $raw_post['wp_post_id'],
				];

				if ( isset( $row['wp_ancestors'] ) ) {
					$post['ancestors'] = $row['wp_ancestors'];
				}

				$post['wp_post_meta'] = isset( $raw_post['wp_postmeta'] ) ? array_map( $postmeta_callback, $raw_post['wp_postmeta'] ) : [];
				$post['ld_category']  = isset( $raw_post['ld_categories'] ) ? $raw_post['ld_categories'] : [];

				$posts[ $raw_post['wp_post_id'] ] = $post;
			}

			$this->set_posts( $posts );
		}

		/**
		 * Abstract
		 *
		 * @param array $content_settings .
		 */
		protected function import_global_content_setting( $content_settings ) {
			if ( empty( $content_settings ) ) {
				return null;
			}

			foreach ( $content_settings as $content_setting ) {
				foreach ( $content_setting as $key => $value ) {
					if ( ! empty( $value ) ) {
						update_option( $key, $value );
					}
				}
			}
		}

		/**
		 * Abstract
		 *
		 * @param array $content_settings .
		 */
		protected function import_users_data( $users ) {

			if ( empty( $users ) ) {
				return;
			}

			foreach ( $users as $user ) {

				$user_exist = get_user_by( 'email', $user['user_email'] );
				if ( ! $user_exist ) {
					$user_data = [
						'user_login'          => $user['user_login'],
						'user_pass'           => $user['user_pass'],
						'user_nicename'       => $user['user_nicename'],
						'user_email'          => $user['user_email'],
						'user_url'            => $user['user_url'],
						'user_registered'     => $user['user_registered'],
						'user_activation_key' => $user['user_activation_key'],
						'user_status'         => $user['user_status'],
						'display_name'        => $user['display_name'],
					];
					$user_id =  wp_insert_user( $user_data );
				} else {
					$user_id = $user_exist->ID;
				}
				$user['meta']['ldie_old_user_id']              = $user['ID'];
				$user['meta']['ldie_old_user_id_to_be_import'] = $user['ID'];
				if ( ! is_wp_error( $user_id ) && ! empty( $user['meta'] ) ) {
					foreach ( $user['meta'] as $key => $value ) {
						$value = maybe_unserialize( $value );
						update_user_meta( $user_id, $key, $value );
					}
				}


			}
		}
	}
}
