<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_Process_Actions' ) ) {
	/**
	 * Admin Import
	 */
	class Ld_Admin_Import_Process_Actions {

		const PROCESS_OPTION           = 'ldie_import_data';
		const ANALYZER_ACTION          = 'ldie_import_analyzer';
		const SETTINGS_IMPORT_ACTION   = 'ldie_import_settings';
		const USERS_IMPORT_ACTION      = 'ldie_import_users';
		const CATEGORIES_IMPORT_ACTION = 'ldie_import_categories';
		const QUIZ_PRO_IMPORT_ACTION   = 'ldie_import_quiz_pro';
		const POSTS_IMPORT_ACTION      = 'ldie_import_posts';
		const MAPPER_ACTION            = 'ldie_import_mapper';
		const USER_MAPPER_ACTION       = 'ldie_import_user_mapper';
		const CLEANUP_ACTION           = 'ldie_import_cleaner';
		const ACTION_GROUP             = 'ldie-import-data-group';


		public function add_actions() {
			add_action( self::ANALYZER_ACTION, [ $this, 'run_import_analyzer' ] );
			add_action( self::SETTINGS_IMPORT_ACTION, [ $this, 'run_import_settings' ] );
			add_action( self::USERS_IMPORT_ACTION, [ $this, 'run_import_users' ] );
			add_action( self::CATEGORIES_IMPORT_ACTION, [ $this, 'run_import_categories' ] );
			add_action( self::QUIZ_PRO_IMPORT_ACTION, [ $this, 'run_import_quiz_pro' ] );
			add_action( self::POSTS_IMPORT_ACTION, [ $this, 'run_import_posts' ] );
			add_action( self::MAPPER_ACTION, [ $this, 'run_import_mapper' ] );
			add_action( self::USER_MAPPER_ACTION, [ $this, 'run_import_user_mapper' ] );
			add_action( self::CLEANUP_ACTION, [ $this, 'run_import_cleaner' ] );
		}

		public function run_import_analyzer() {
			$data = $this->get_import_option();

			if ( ! isset( $data['process'] ) ) {
				$data['process'] = [
					'analyze'    => [
						'status'         => 'in-progress',
						'keys'           => [
							'global_settings',
							'users_data',
							'categories',
							'quiz_pro',
							'posts',
						],
						'analyzed_files' => 0,
						'total_files'    => count( $data['files'] ),
					],
					'global_settings' => [
						'status'    => 'pending',
						'imported'  => 0,
						'to_import' => 0,
						'files'     => [],
					],
					'users_data'      => [
						'status'    => 'pending',
						'imported'  => 0,
						'to_import' => 0,
						'files'     => [],
					],
					'categories'      => [
						'status'    => 'pending',
						'imported'  => 0,
						'to_import' => 0,
						'files'     => [],
					],
					'quiz_pro'        => [
						'status'    => 'pending',
						'imported'  => 0,
						'to_import' => 0,
						'files'     => [],
					],
					'posts'           => [
						'status'    => 'pending',
						'imported'  => 0,
						'to_import' => 0,
						'files'     => [],
					],
					'mapper'          => [
						'status' => 'pending',
						'mapped' => 0,
						'to_map' => 0,
					],
					'user_mapper'     => [
						'status' => 'pending',
						'mapped' => 0,
						'to_map' => 0,
					],
				];
			}

			if ( 'complete' === $data['process']['analyze']['status'] ) {
				$this->cancel_action_if_exists( self::ANALYZER_ACTION );
				if ( false === as_next_scheduled_action( self::SETTINGS_IMPORT_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::SETTINGS_IMPORT_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			if ( 0 === count( $data['process']['analyze']['keys'] ) ) {
				$data['process']['analyze']['status'] = 'complete';
				update_option( self::PROCESS_OPTION, $data );
				return;
			}

			$current_key    = $data['process']['analyze']['keys'][0];
			$current_files  = array_column( $data['process'][ $current_key ]['files'], 'filename' ) ?: [];
			$files_to_count = $this->get_files_by_pattern( $data['files'], $current_key, $current_files );

			if ( 0 === count( $files_to_count ) ) {
				array_shift( $data['process']['analyze']['keys'] );
				update_option( self::PROCESS_OPTION, $data );
				return;
			}

			$first = array_shift( $files_to_count );
			$count = $this->get_count_by_file( $first, $data['dir'], $current_key );

			$data['process'][ $current_key ]['to_import'] += $count;

			if ( 'posts' === $current_key ) {
				$data['process']['mapper']['to_map'] += $count;
			}

			if ( 'users_data' === $current_key ) {
				$data['process']['user_mapper']['to_map'] += $count;
			}

			$data['process'][ $current_key ]['files'][]    = [
				'filename' => $first,
				'status'   => 'loading',
			];
			$data['process']['analyze']['analyzed_files'] += 1;

			update_option( self::PROCESS_OPTION, $data );
		}

		private function get_files_by_pattern( $files, $pattern, $exclude = [] ) {
			$matches = [];

			if ( 'posts' === $pattern ) {
				$pattern = 'groups|sfwd-|attachments';
			}

			foreach ( $files as $file ) {

				if ( in_array( $file, $exclude, true ) ) {
					continue;
				}

				if ( preg_match( "@{$pattern}@", $file ) ) {
					array_push( $matches, $file );
				}
			}

			return $matches;
		}

		private function get_importer( $file, $dir ) {

			if ( ! preg_match( '/\.(json|xml)$/', $file, $extension_match ) ) {
				return;
			}

			if ( 'json' === $extension_match[1] ) {
				return new Ld_Admin_Import_JSON_File();
			}

			if ( 'xml' === $extension_match[1] ) {
				return new Ld_Admin_Import_XML_File();
			}

			return null;
		}


		private function get_count_by_file( $file, $dir, $key ) {
			$filepath = Ld_Admin_File_System::get_wp_upload_dir( $dir . '/' . $file );
			$importer = $this->get_importer( $file, $dir );

			if ( ! $importer ) {
				return 0;
			}

			return $importer->count_data_from_file( $filepath, $key );
		}

		public function run_import_settings() {
			$this->cancel_action_if_exists( self::ANALYZER_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['global_settings']['status'] ) {
				$data['process']['global_settings']['status'] = 'in-progress';
			}

			if ( 'complete' === $data['process']['global_settings']['status'] ) {
				$this->cancel_action_if_exists( self::SETTINGS_IMPORT_ACTION );
				if ( false === as_next_scheduled_action( self::USERS_IMPORT_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::USERS_IMPORT_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$this->import_file_by_key( $data, 'global_settings' );
		}

		public function run_import_users() {
			$this->cancel_action_if_exists( self::SETTINGS_IMPORT_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['users_data']['status'] ) {
				$data['process']['users_data']['status'] = 'in-progress';
			}

			if ( 'complete' === $data['process']['users_data']['status'] ) {
				$this->cancel_action_if_exists( self::USERS_IMPORT_ACTION );
				if ( false === as_next_scheduled_action( self::CATEGORIES_IMPORT_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::CATEGORIES_IMPORT_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$this->import_file_by_key( $data, 'users_data' );
		}

		public function run_import_categories() {
			$this->cancel_action_if_exists( self::USERS_IMPORT_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['categories']['status'] ) {
				$data['process']['categories']['status'] = 'in-progress';
			}

			if ( 'complete' === $data['process']['categories']['status'] ) {
				$this->cancel_action_if_exists( self::CATEGORIES_IMPORT_ACTION );
				if ( false === as_next_scheduled_action( self::QUIZ_PRO_IMPORT_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::QUIZ_PRO_IMPORT_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$this->import_file_by_key( $data, 'categories' );
		}

		public function run_import_quiz_pro() {
			$this->cancel_action_if_exists( self::CATEGORIES_IMPORT_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['quiz_pro']['status'] ) {
				$data['process']['quiz_pro']['status'] = 'in-progress';
			}

			if ( 'complete' === $data['process']['quiz_pro']['status'] ) {
				$this->cancel_action_if_exists( self::QUIZ_PRO_IMPORT_ACTION );
				if ( false === as_next_scheduled_action( self::POSTS_IMPORT_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::POSTS_IMPORT_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$this->import_file_by_key( $data, 'quiz_pro' );
		}

		public function run_import_posts() {
			$this->cancel_action_if_exists( self::QUIZ_PRO_IMPORT_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['posts']['status'] ) {
				$data['process']['posts']['status'] = 'in-progress';
			}

			if ( 'complete' === $data['process']['posts']['status'] ) {
				$this->cancel_action_if_exists( self::POSTS_IMPORT_ACTION );
				if ( false === as_next_scheduled_action( self::MAPPER_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::MAPPER_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$this->import_file_by_key( $data, 'posts' );
		}

		private function import_file_by_key( $data, $key ) {
			$files          = $data['process'][ $key ]['files'];
			$slow_run       = $data['settings']['is_slow_run'];
			$filtered_files = [];

			foreach ( $files as $file ) {
				if ( 'loading' === $file['status'] ) {
					$filtered_files[] = $file['filename'];
				}
			}

			if ( 0 === count( $filtered_files ) ) {
				$data['process'][ $key ]['status'] = 'complete';
				update_option( self::PROCESS_OPTION, $data );
				return;
			}

			$filepath = Ld_Admin_File_System::get_wp_upload_dir( $data['dir'] . '/' . $filtered_files[0] );
			$importer = $this->get_importer( $filtered_files[0], $data['dir'] );

			if ( $slow_run['enabled'] && 'categories' !== $key ) {
				$result = $importer->run( $filepath, $slow_run['context']['imported_count'], $slow_run['context']['batch_to_run'] );

				$slow_run['context']['total_count']     = $result['total_count'];
				$slow_run['context']['imported_count'] += $result['imported_count'];
				$data['process'][ $key ]['imported']   += $result['imported_count'];
				if ( $slow_run['context']['imported_count'] >= $slow_run['context']['total_count'] ) {
					$found_key = array_search( $filtered_files[0], array_column( $files, 'filename' ), true );
					$data['process'][ $key ]['files'][ $found_key ]['status'] = $result['success'] ? 'complete' : 'error';

					$slow_run['context']['imported_count'] = 0;
					$slow_run['context']['total_count']    = 0;
				}

				$data['settings']['is_slow_run'] = $slow_run;
			} else {
				$result    = $importer->run( $filepath );
				$found_key = array_search( $filtered_files[0], array_column( $files, 'filename' ), true );
				$data['process'][ $key ]['files'][ $found_key ]['status'] = $result['success'] ? 'complete' : 'error';
				$data['process'][ $key ]['imported']                     += $result['imported_count'];
			}

			if ( $data['process'][ $key ]['imported'] === $data['process'][ $key ]['to_import'] ) {
				$data['process'][ $key ]['status'] = 'complete';
			}

			update_option( self::PROCESS_OPTION, $data );
		}

		public function run_import_mapper() {
			$this->cancel_action_if_exists( self::POSTS_IMPORT_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['mapper']['status'] ) {
				$data['process']['mapper']['status'] = 'in-progress';
			}

			// on completed.
			if ( 'complete' === $data['process']['mapper']['status'] ) {
				$this->cancel_action_if_exists( self::MAPPER_ACTION );
				if ( false === as_next_scheduled_action( self::USER_MAPPER_ACTION, [], self::ACTION_GROUP ) ) {
					as_schedule_recurring_action( strtotime( 'now' ) + 10, 3, self::USER_MAPPER_ACTION, [], self::ACTION_GROUP );
				}
				return;
			}

			$importer = new Ld_Admin_Import_Common_File();
			$result   = $importer->run_post_metas_mapper();

			$data['process']['mapper']['mapped'] += $result['count_post_mapped'];

			if ( $result['is_last_batch'] ) {
				$data['process']['mapper']['status'] = 'complete';
			}
			update_option( self::PROCESS_OPTION, $data );
		}

		public function run_import_user_mapper() {
			$this->cancel_action_if_exists( self::MAPPER_ACTION );
			$data = $this->get_import_option();

			if ( 'pending' === $data['process']['user_mapper']['status'] ) {
				$data['process']['user_mapper']['status'] = 'in-progress';
			}

			// on completed
			if ( 'complete' === $data['process']['user_mapper']['status'] ) {
				$this->cancel_action_if_exists( self::USER_MAPPER_ACTION );
				$data['completed'] = true;
				update_option( self::PROCESS_OPTION, $data );
				return;
			}

			$importer = new Ld_Admin_Import_Common_File();
			$result   = $importer->run_user_metas_mapper();

			$data['process']['user_mapper']['mapped'] += $result['count_user_mapped'];

			if ( $result['is_last_batch'] ) {
				$data['process']['user_mapper']['status'] = 'complete';
			}
			update_option( self::PROCESS_OPTION, $data );
		}

		public function run_import_cleaner() {
			// at this point, the get_status checker will be stopped.
			$this->cancel_action_if_exists( self::ANALYZER_ACTION );
			$this->cancel_action_if_exists( self::SETTINGS_IMPORT_ACTION );
			$this->cancel_action_if_exists( self::USERS_IMPORT_ACTION );
			$this->cancel_action_if_exists( self::CATEGORIES_IMPORT_ACTION );
			$this->cancel_action_if_exists( self::QUIZ_PRO_IMPORT_ACTION );
			$this->cancel_action_if_exists( self::POSTS_IMPORT_ACTION );
			$this->cancel_action_if_exists( self::MAPPER_ACTION );
			$this->cancel_action_if_exists( self::USER_MAPPER_ACTION );

			$data = $this->get_import_option();

			if ( ! empty( $data ) ) {
				Ld_Admin_File_System::delete_import_dir( $data['dir'] );
			}

			delete_option( self::PROCESS_OPTION );
		}

		private function get_import_option() {
			return get_option( self::PROCESS_OPTION, [] );
		}

		private function cancel_action_if_exists( $action ) {
			as_unschedule_all_actions( $action, [], self::ACTION_GROUP );
			as_unschedule_action( $action, [], self::ACTION_GROUP );
		}
	}
}
