<?php
/**
 * LDIE import import class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Import_XML_File' ) ) {

	/**
	 * Admin Import
	 */
	class Ld_Admin_Import_XML_File extends Ld_Admin_Import_Abstract_File {

		/**
		 * Abstract
		 *
		 * @param string $filepath .
		 * @return mixed
		 */
		protected function prepare_file_data( $filepath ) {
			$xml = simplexml_load_file( $filepath, null, LIBXML_NOCDATA );

			if ( ! $this->is_valid_data( $xml ) ) {
				throw new \Exception( esc_html__( 'Corrupted or Invalid data', 'ea_import_export' ) );
			}

			return $xml;
		}

		/**
		 * Undocumented function
		 *
		 * @param object $xml .
		 * @return boolean
		 */
		protected function is_valid_data( $xml ) {
			return md5( 'ldie' ) === (string) $xml->transkey->keyid[0];
		}

		/**
		 * Abstract function to initialize the structure
		 *
		 * @param array $data .
		 * @return array
		 */
		protected function init( $data ) {

			$xml_array               = (array) $data;
			$init_data               = [];
			$init_data['categories'] = $this->parse_course_categories( isset( $xml_array['wp_category'] ) ? $xml_array['wp_category'] : [] );
			$init_data['quiz_pro']   = [
				'questions'  => $this->parse_quiz_pro_questions( isset( $xml_array['quiz_pro_questions'] ) ? $xml_array['quiz_pro_questions'] : [] ),
				'categories' => $this->parse_quiz_pro_categories( isset( $xml_array['quiz_pro_categories'] ) ? $xml_array['quiz_pro_categories'] : [] ),
			];
			$init_data['posts']      = isset( $xml_array['item'] ) ? $xml_array['item'] : [];

			return $init_data;
		}

		/**
		 * Undocumented function
		 *
		 * @param array $init_categories .
		 * @return void
		 */
		protected function import_categories( $init_categories ) {
			$categories = [];
			foreach ( $init_categories as $category ) {
				$categories[] = [
					'term_id'         => $category['wp_term_id'],
					'term_taxonomy'   => $category['wp_term_taxonomy'],
					'slug'            => $category['wp_category_nicename'],
					'category_parent' => $category['wp_category_parent'],
					'name'            => $category['wp_cat_name'],
				];
			}
			if ( ! empty( $categories ) ) {
				$this->set_categories( $categories );
			}
		}

		/**
		 * Create the Posts
		 *
		 * @param object $raw_posts .
		 * @return void
		 */
		protected function import_posts( $raw_posts ) {
			$posts = [];

			$postmeta_callback = function( $meta ) {
				$as_array = (array) $meta;
				return [
					'meta_key'   => trim( (string) $as_array['wp_meta_key'] ), // phpcs:ignore
					'meta_value' => trim( (string) $as_array['wp_meta_value'] ), // phpcs:ignore
				];
			};

			$ldcategory_callback = function( $category ) {
				$as_array = (array) $category;
				return [
					'ld_slug'   => (string) $as_array['ld_slug'], // phpcs:ignore
					'ld_category_name' => (string) $as_array['ld_cat_name'], // phpcs:ignore
					'ld_taxonomy' => (string) $as_array['ld_taxonomy'], // phpcs:ignore
				];
			};

			if ( ! is_array( $raw_posts ) ) {
				$raw_posts = [ $raw_posts ];
			}

			foreach ( $raw_posts as $index => $raw_post ) {
				$row     = (array) $raw_post;
				$content = '';

				if ( ! is_object( $row['content_encoded'] ) ) {
					$content = $this->replace_tokens_on_content( $row['content_encoded'] );
				}
				$post = [
					'post_author'           => ( ! is_object( $row['dc_creator'] ) ) ? $row['dc_creator'] : '',
					'post_date'             => ( ! is_object( $row['wp_post_date'] ) ) ? $row['wp_post_date'] : '',
					'post_date_gmt'         => ( ! is_object( $row['wp_post_date_gmt'] ) ) ? $row['wp_post_date_gmt'] : '',
					'post_content'          => $content,
					'post_content_filtered' => wp_strip_all_tags( $content ),
					'post_title'            => ( ! is_object( $row['wp_post_title'] ) ) ? $row['wp_post_title'] : '',
					'post_excerpt'          => ( ! is_object( $row['excerpt_encoded'] ) ) ? $row['excerpt_encoded'] : '',
					'post_status'           => ( ! is_object( $row['wp_post_status'] ) ) ? $row['wp_post_status'] : '',
					'post_type'             => ( ! is_object( $row['wp_post_type'] ) ) ? $row['wp_post_type'] : '',
					'comment_status'        => ( ! is_object( $row['wp_comment_status'] ) ) ? $row['wp_comment_status'] : '',
					'ping_status'           => ( ! is_object( $row['wp_ping_status'] ) ) ? $row['wp_ping_status'] : '',
					'post_password'         => ( ! is_object( $row['wp_post_password'] ) ) ? $row['wp_post_password'] : '',
					'post_name'             => ( ! is_object( $row['wp_post_name'] ) ) ? $row['wp_post_name'] : '',
					'to_ping'               => ( ! is_object( $row['wp_to_ping'] ) ) ? $row['wp_to_ping'] : '',
					'pinged'                => ( ! is_object( $row['wp_pinged'] ) ) ? $row['wp_pinged'] : '',
					'post_modified'         => ( ! is_object( $row['wp_post_modified'] ) ) ? $row['wp_post_modified'] : '',
					'post_modified_gmt'     => ( ! is_object( $row['wp_post_modified_gmt'] ) ) ? $row['wp_post_modified_gmt'] : '',
					'post_parent'           => ( ! is_object( $row['wp_post_parent'] ) ) ? $row['wp_post_parent'] : '',
					'menu_order'            => ( ! is_object( $row['wp_menu_order'] ) ) ? $row['wp_menu_order'] : '',
					'post_mime_type'        => ( ! is_object( $row['wp_post_mime_type'] ) ) ? $row['wp_post_mime_type'] : '',
					'guid'                  => ( ! is_object( $row['guid'] ) ) ? $row['guid'] : '',
				];

				if ( isset( $row['wp_ancestors'] ) && ! is_object( $row['wp_ancestors'] ) ) {
					$post['ancestors'] = json_decode( $row['wp_ancestors'] );
				}

				$post['wp_post_meta'] = isset( $row['wp_postmeta'] ) ? array_map( $postmeta_callback, $row['wp_postmeta'] ) : [];
				$post['ld_category']  = isset( $row['ld_category'] ) ? array_map( $ldcategory_callback, $row['ld_category'] ) : [];

				$posts[ $row['wp_post_id'] ] = $post;
			}

			$this->set_posts( $posts );
		}


		private function replace_tokens_on_content( $content ) {
			$content = str_replace( '{ldie_line_break}', "\n", $content );
			return $content;
		}

		/**
		 * Parse the questions from XML to structured data
		 *
		 * @param array $raw_questions .
		 * @return array
		 */
		private function parse_quiz_pro_questions( $raw_questions ) {

			if ( ! empty( $raw_questions->q_pro_question ) ) {

				$questions = [];

				foreach ( $raw_questions->q_pro_question as $question ) {
					$questions[] = unserialize( $question );
				}

				return $questions;
			}

			if ( ! empty( $raw_questions ) ) {
				$legacy_unserielized = unserialize( $raw_questions );
				if ( false !== $legacy_unserielized ) {
					return $legacy_unserielized;
				}
			}

			return [];
		}

		/**
		 * Parse the categories from XML Object to Array.
		 *
		 * @param array $raw_categories Category and Tag Data.
		 *
		 * @return array $categories
		 */
		private function parse_course_categories( $raw_categories ) {
			$categories = [];

			if ( empty( $raw_categories ) && is_array( $raw_categories ) ) {
				return $categories;
			}

			foreach ( $raw_categories as $category ) {
				$categories[] = [
					'wp_term_id'           => (string) $category->wp_term_id,
					'wp_term_taxonomy'     => (string) $category->wp_term_taxonomy,
					'wp_category_nicename' => (string) $category->wp_category_nicename,
					'wp_category_parent'   => (string) $category->wp_category_parent,
					'wp_cat_name'          => (string) $category->wp_cat_name,
				];
			}

			return $categories;
		}

		/**
		 * Parse the questions categories from XML to structured data
		 *
		 * @param array $raw_categories .
		 * @return array
		 */
		private function parse_quiz_pro_categories( $raw_categories ) {
			$categories = [];

			if ( empty( $raw_categories ) && is_array( $raw_categories ) ) {
				return $categories;
			}

			foreach ( $raw_categories->category as $category ) {
				$categories[] = [
					'id'   => (string) $category->id,
					'name' => (string) $category->name,
				];
			}

			return $categories;
		}

		/**
		 * Abstract
		 *
		 * @param array $settings .
		 */
		protected function import_global_content_setting( $settings ) {
			// Todo...
		}

		/**
		 * Abstract
		 */
		protected function import_users_data( $users ) {
			// Todo...
		}
	}
}
