<?php
/**
 * LDIE admin page hook class
 *
 * @since 1.0.0
 * @package ea-import-export/admin/includes
 * Contributor: Jonatan Treviño
 */

namespace LDIE\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Ld_Admin_Reset_Data' ) ) {

	/**
	 * Ld_Admin_Reset_Data class
	 */
	class Ld_Admin_Reset_Data {


		protected $text;

		public function __construct() {
			$this->text = Ld_Text::text( 'notices' );
			add_action( 'wp_ajax_ldie_reset_data', [ $this, 'do_call' ] );
			add_action( 'wp_ajax_ldie_reset_status', [ $this, 'get_status' ] );
			add_action( 'wp_ajax_ldie_reset_cancel_action', [ $this, 'cancel_action' ] );
			add_action( 'ldie_reset_all_ld_data', [ $this, 'run' ] );
		}

		public function cancel_action() {
			try {

				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_reset_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				if ( false !== as_next_scheduled_action( 'ldie_reset_all_ld_data', [], 'ldie-reset-data' ) ) {
					as_unschedule_action( 'ldie_reset_all_ld_data', [], 'ldie-reset-data' );
					as_unschedule_all_actions( 'ldie_reset_all_ld_data', [], 'ldie-reset-data' );
				}

				echo wp_json_encode(
					[
						'status' => 'success',
					]
				);
			} catch ( \Exception $error ) {
				echo wp_json_encode(
					[
						'status' => 'error',
					]
				);
			}
			exit();
		}

		public function get_status() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_reset_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$attempts   = isset( $_POST['attempts'] ) ? sanitize_text_field( wp_unslash( $_POST['attempts'] ) ) : 0;
				$reset_data = get_option(
					'ldie_reset_data',
					[
						'status'     => 'loading',
						'post_types' => [],
						'quiz_pro'   => [],
					]
				);

				switch ( $reset_data['status'] ) {
					case 'complete':
						$message = 'Data is successfully deleted';
						break;
					case 'failed':
						$message = 'Unexpected Error.';
						break;
					case 'pending':
					default:
						$message = '';
						break;
				}

				echo wp_json_encode(
					[
						'status'       => 'success',
						'reset_status' => $reset_data['status'],
						'message'      => $message,
					]
				);

			} catch ( \Exception $error ) {
				echo wp_json_encode(
					[
						'status'  => 'error',
						'data'    => 'failed',
						'message' => $error->getMessage(),
					]
				);
			}
			exit();
		}


		public function do_call() {
			try {
				if ( isset( $_POST['learndash_settings_import_export_nonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['learndash_settings_import_export_nonce'] ) ), 'learndash_reset_data' ) ) {
					throw new \Exception( $this->text['nonce'] );
				}

				if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
					throw new \Exception( esc_html__( 'Not Allowed', 'ea_import_export' ) );
				}

				$post_types = [
					'sfwd-courses',
					'sfwd-lessons',
					'sfwd-topic',
					'sfwd-quiz',
					'sfwd-question',
					'sfwd-assignment',
					'sfwd-certificates',
					'groups',
				];

				$quiz_pro = [
					'category',
					'question',
					'master',
				];

				$post_types = array_filter(
					$post_types,
					function ( $post_type ) {
						return 0 !== $this->get_post_type_count( $post_type );
					}
				);

				$quiz_pro = array_filter(
					$quiz_pro,
					function ( $quiz_pro_table ) {
						return 0 !== $this->get_quiz_pro_count( $quiz_pro_table );
					}
				);

				$status = 'loading';

				if ( empty( $quiz_pro ) && empty( $post_types ) ) {
					$status = 'complete';
				} else {
					as_schedule_recurring_action( strtotime( 'now' ) + 2, 3, 'ldie_reset_all_ld_data', [], 'ldie-reset-data' );
				}

				update_option(
					'ldie_reset_data',
					[
						'status'     => $status,
						'post_types' => array_values( $post_types ),
						'quiz_pro'   => array_values( $quiz_pro ),
					]
				);

				echo wp_json_encode(
					[
						'status' => 'success',
					]
				);
			} catch ( \Exception $e ) {
				wp_send_json_error(
					[
						'status'  => 'error',
						'message' => $e->getMessage(),
					],
					500
				);
			}
			exit();
		}

		protected function get_post_type_count( $post_type ) {
			global $wpdb;

			$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->prefix}posts WHERE post_type = %s", $post_type ) );

			return $count;
		}


		protected function get_quiz_pro_count( $quiz_pro_table ) {
			global $wpdb;

			$table = $this->get_quiz_pro_database_table( $quiz_pro_table );

			$count = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

			return $count;
		}

		public function run() {

			$reset_option = get_option(
				'ldie_reset_data',
				[
					'status'     => 'loading',
					'post_types' => [],
					'quiz_pro'   => [],
				]
			);

			if ( ! empty( $reset_option['post_types'] ) ) {
				$deleted_count = $this->delete_posts_by_single_post_type( $reset_option['post_types'][0] );

				if ( 0 === $deleted_count ) {
					array_shift( $reset_option['post_types'] );
					update_option( 'ldie_reset_data', $reset_option );
					return;
				}
			}

			if ( ! empty( $reset_option['quiz_pro'] ) ) {
				$table = array_shift( $reset_option['quiz_pro'] );
				$this->delete_quiz_pro_data( $table );
				update_option( 'ldie_reset_data', $reset_option );

				if ( 0 === count( $reset_option['quiz_pro'] ) ) {
					delete_option( 'ldie_quiz_categories_map' );
					delete_option( 'ldie_quiz_questions_map' );
					delete_option( 'ldie_old_quiz_new_question_map' );
				}
			}

			if ( empty( $reset_option['post_types'] ) && empty( $reset_option['quiz_pro'] ) ) {
				$reset_option['status'] = 'complete';
				update_option( 'ldie_reset_data', $reset_option );
			}
		}

		private function delete_posts_by_single_post_type( $post_type ) {

			$args = [
				'posts_per_page' => 50,
				'post_type'      => $post_type,
				'post_status'    => 'any',
			];

			$posts = get_posts( $args );

			foreach ( $posts as $data ) {
				wp_delete_post( $data->ID, true );
			}

			return count( $posts );
		}

		private function get_quiz_pro_database_table( $type ) {
			global $wpdb;

			$table_name = "{$wpdb->prefix}learndash_pro_quiz_{$type}";

			if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) === $table_name ) {
				return $table_name;
			}

			return "{$wpdb->prefix}wp_pro_quiz_{$type}";
		}


		private function delete_quiz_pro_data( $table ) {
			global $wpdb;

			$question_table = $this->get_quiz_pro_database_table( $table );
			$wpdb->query( "DELETE FROM {$question_table}" );
		}
	}
}
