<?php

namespace uncanny_learndash_reporting;

/**
 * Class Boot
 */
class Boot extends Config {

	/**
	 * Class constructor
	 */
	public function __construct() {

		global $uncanny_learndash_reporting;
		$visibility_option = get_option( '_uncanny_tin_canny_try_automator_visibility' );

		// Check if the user chose to hide it.
		if ( empty( $visibility_option ) ) {
			// Register the endpoint to hide the "Try Automator".
			add_action(
				'rest_api_init',
				function () {
					/**
					 * Method try_automator_rest_register.
					 *
					 * Callback method to action hook `rest_api_init`.
					 *
					 * Registers a REST API endpoint to change the visibility of the "Try Automator" item.
					 *
					 * @since 3.5.4
					 */

					register_rest_route(
						'uncanny_reporting/v1',
						'/try_automator_visibility/',
						array(
							'methods'             => 'POST',
							'callback'            => function ( $request ) {

								// Check if its a valid request.
								$data = $request->get_params();

								if ( isset( $data['action'] ) && ( 'hide-forever' === $data['action'] || 'hide-forever' === $data['action'] ) ) {

									update_option( '_uncanny_tin_canny_try_automator_visibility', $data['action'] );

									return new \WP_REST_Response( array( 'success' => true ), 200 );

								}

								return new \WP_REST_Response( array( 'success' => false ), 200 );

							},
							'permission_callback' => function () {
								return true;
							},
						)
					);
				},
				99
			);
		}
		if ( ! isset( $uncanny_learndash_reporting ) ) {
			$uncanny_learndash_reporting = new \stdClass();
		}

		// We need to check if spl auto loading is available when activating plugin
		// Plugin will not activate if SPL extension is not enabled by throwing error
		if ( ! extension_loaded( 'SPL' ) ) {
			trigger_error( esc_html__( 'Please contact your hosting company to update to php version 5.3+ and enable spl extensions.', 'uncanny-learndash-reporting' ), E_USER_ERROR ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_trigger_error
		}

		spl_autoload_register( array( __CLASS__, 'auto_loader' ) );

		$uncanny_learndash_reporting->admin_menu               = new ReportingAdminMenu();
		$uncanny_learndash_reporting->reporting_api            = new ReportingApi();
		$uncanny_learndash_reporting->quiz_module_reports      = new QuizModuleReports();
		$uncanny_learndash_reporting->question_analysis_report = new QuestionAnalysisReport();
		$uncanny_learndash_reporting->lesson_topic_reports     = new LessonTopicReports();
		$uncanny_learndash_reporting->group_quiz_report        = new GroupQuizReport();
		$uncanny_learndash_reporting->tincanny_zip_uploader    = new TincannyZipUploader();

		// URL of store powering the plugin
		define( 'UO_REPORTING_STORE_URL', 'https://licensing.uncannyowl.com/' ); // you should use your own CONSTANT name, and be sure to replace it throughout this file

		// Store download name/title
		define( 'UO_REPORTING_ITEM_NAME', 'Tin Canny LearnDash Reporting' ); // you should use your own CONSTANT name, and be sure to replace it throughout this file
		define( 'UO_REPORTING_ITEM_ID', 4113 ); // you should use your own CONSTANT name, and be sure to replace it throughout this file

		add_action( 'admin_init', array( __CLASS__, 'uo_reporting_register_option' ) );
		add_action( 'admin_menu', array( __CLASS__, 'add_help_submenu' ), 31 );
		add_action( 'admin_menu', array( __CLASS__, 'add_checkpage_submenu' ), 33 );
		add_action( 'admin_menu', array( __CLASS__, 'add_uncanny_plugins_page' ), 32 );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_external_scripts' ) );
		add_action( 'admin_init', array( __CLASS__, 'uo_admin_help_process' ) );
		/* Licensing */
		// Setup menu and page options in admin
		if ( is_admin() ) {

			// Licensing is not autoloaded, load manually
			include_once self::get_include( 'licensing.php' );

			// Create a new instance of EDD Liccensing
			$licensing = new Licensing();

			// Create sub-page for EDD licensing
			$licensing->page_name   = 'Uncanny Reporting License Activation';
			$licensing->page_slug   = 'uncanny-reporting-license-activation';
			$licensing->parent_slug = 'uncanny-learnDash-reporting';
			$licensing->store_url   = UO_REPORTING_STORE_URL;
			$licensing->item_name   = UO_REPORTING_ITEM_NAME;
			$licensing->item_id     = UO_REPORTING_ITEM_ID;
			$licensing->author      = 'Uncanny Owl';
			$licensing->add_licensing();

		}

		// Check if the protection is enabled
		if ( get_option( 'tincanny_nonce_protection', 'yes' ) === 'yes' ) {
			self::create_protection_htaccess();
		}

		if ( get_option( 'tincanny_snc_dir_permissions', 'no' ) === 'no' ) {
			self::update_snc_dir_permissions();
		}
	}

	/**
	 * Create .htaccess file in the uncanny-snc folder
	 *
	 * @return void
	 */
	public static function create_protection_htaccess() {
		// Check if the constant with the name of the Tin Canny folder is defined
		if ( ! defined( 'SnC_UPLOAD_DIR_NAME' ) ) { // phpcs:ignore Generic.NamingConventions
			// If it's not, then define it
			define( 'SnC_UPLOAD_DIR_NAME', 'uncanny-snc' ); // phpcs:ignore Generic.NamingConventions
		}

		$wp_upload_dir = wp_upload_dir();
		$upload_dir    = $wp_upload_dir['basedir'] . '/' . SnC_UPLOAD_DIR_NAME;

		if ( file_exists( $upload_dir ) ) {
			if ( ! file_exists( $upload_dir . '/.htaccess' ) ) {
				if ( defined( 'UO_ABS_PATH' ) ) {

					require_once ABSPATH . 'wp-admin/includes/file.php';
					global $wp_filesystem;
					\WP_Filesystem();

					$slashed_home = trailingslashit( get_option( 'home' ) );
					$base         = wp_parse_url( $slashed_home, PHP_URL_PATH );

					$htaccess_file = <<<EOF
<IfModule mod_rewrite.c>
RewriteEngine On
RewriteBase {$base}
RewriteRule ^index\.php$ - [L]
RewriteRule ^(?:|(?:\/|\\\\))([0-9]{1,})((?:.*(?:\/|\\\\))|.*\.(?:(?:html|htm)(?:|.*)))$ {$base}index.php?tincanny_content_id=$1&tincanny_file_path=$2 [QSA,L,NE]
</IfModule>
EOF;

					$wp_filesystem->put_contents( $upload_dir . '/.htaccess', $htaccess_file );
				}
			}
		}
	}

	/**
	 * Delete .htaccess file in the uncanny-snc folder
	 *
	 * @return void
	 */
	public static function delete_protection_htaccess() {
		// Check if the constant with the name of the Tin Canny folder is defined
		if ( ! defined( 'SnC_UPLOAD_DIR_NAME' ) ) { // phpcs:ignore Generic.NamingConventions
			// If it's not, then define it
			define( 'SnC_UPLOAD_DIR_NAME', 'uncanny-snc' ); // phpcs:ignore Generic.NamingConventions
		}

		// Get the upload directory (uncanny-snc folder)
		$wp_upload_dir = wp_upload_dir();
		$upload_dir    = $wp_upload_dir['basedir'] . '/' . SnC_UPLOAD_DIR_NAME;

		// Check if the folder exists
		if ( file_exists( $upload_dir ) ) {
			// Check if the .htaccess was created in the uncanny-snc folder
			if ( file_exists( $upload_dir . '/.htaccess' ) ) {
				// Require file.php. Use require_once to avoid including it again
				// if it's already there
				require_once ABSPATH . 'wp-admin/includes/file.php';
				// Get global wp_filesystem
				global $wp_filesystem;
				// Create instance of WP_Filesystem
				\WP_Filesystem();

				// Remove the file
				$wp_filesystem->delete( $upload_dir . '/.htaccess' );
			}
		}
	}

	/**
	 * Update the permissions of the uncanny-snc folder
	 *
	 * @return void
	 */
	public static function update_snc_dir_permissions() {
		// Check if the constant with the name of the Tin Canny folder is defined
		if ( ! defined( 'SnC_UPLOAD_DIR_NAME' ) ) { // phpcs:ignore Generic.NamingConventions
			// If it's not, then define it
			define( 'SnC_UPLOAD_DIR_NAME', 'uncanny-snc' ); // phpcs:ignore Generic.NamingConventions
		}

		// Get the upload directory (uncanny-snc folder)
		$wp_upload_dir = wp_upload_dir();
		$upload_dir    = $wp_upload_dir['basedir'] . '/' . SnC_UPLOAD_DIR_NAME;

		if ( file_exists( $upload_dir ) ) {
			// Check current permissions of the directory.
			$dir_perms = substr( sprintf( '%o', fileperms( $upload_dir ) ), - 4 );
			// If the permissions are not 0755, change them.
			if ( '0755' !== $dir_perms ) {
				$result = @chmod( $upload_dir, 0755 );
			}
		}
		update_option( 'tincanny_snc_dir_permissions', 'yes' );
	}

	/**
	 * Add "Help" submenu
	 */
	public static function add_help_submenu() {
		add_submenu_page(
			'uncanny-learnDash-reporting',
			__( 'Tin Canny Reporting for LearnDash Support', 'uncanny-learndash-reporting' ),
			__( 'Help', 'uncanny-learndash-reporting' ),
			'manage_options',
			'uncanny-tincanny-kb',
			array( __CLASS__, 'include_help_page' )
		);
	}

	/**
	 * Create "Uncanny Plugins" submenu
	 */
	public static function add_uncanny_plugins_page() {
		add_submenu_page(
			'uncanny-learnDash-reporting',
			__( 'Uncanny LearnDash Plugins', 'uncanny-learndash-reporting' ),
			__( 'LearnDash Plugins', 'uncanny-learndash-reporting' ),
			'manage_options',
			'uncanny-tincanny-plugins',
			array( __CLASS__, 'include_learndash_plugins_page' )
		);
	}

	/**
	 * Add "Check Page" submenu
	 */
	public static function add_checkpage_submenu() {
		add_submenu_page(
			'uncanny-learnDash-reporting',
			__( 'Tin Canny Reporting for LearnDash Support', 'uncanny-learndash-reporting' ),
			__( 'Site check', 'uncanny-learndash-reporting' ),
			'manage_options',
			'uncanny-tincanny-site-check',
			array( __CLASS__, 'include_site_check_page' )
		);
	}

	/**
	 * Include "Help" template
	 */
	public static function include_help_page() {
		include 'templates/admin-help.php';
	}

	/**
	 * Include "Help" template
	 */
	public static function include_site_check_page() {
		include 'templates/admin-site-check.php';
	}

	/**
	 * Include "LearnDash Plugins" template
	 */
	public static function include_learndash_plugins_page() {
		include 'templates/admin-learndash-plugins.php';
	}

	/**
	 * Enqueue external scripts from uncannyowl.com
	 */
	public static function enqueue_external_scripts() {
		$pages_to_include = array( 'uncanny-tincanny-plugins', 'uncanny-tincanny-kb', 'uncanny-tincanny-site-check' );
		$page             = ultc_filter_has_var( 'page' ) ? ultc_filter_input( 'page' ) : false;
		if ( $page && in_array( $page, $pages_to_include, true ) ) {
			wp_enqueue_style( 'uncannyowl-core', 'https://uncannyowl.com/wp-content/mu-plugins/uncanny-plugins-core/dist/bundle.min.css', array(), Config::get_version() );
			wp_enqueue_script( 'uncannyowl-core', 'https://uncannyowl.com/wp-content/mu-plugins/uncanny-plugins-core/dist/bundle.min.js', array( 'jquery' ), Config::get_version(), false );

			wp_enqueue_style( 'tclr-icons', Config::get_admin_css( 'icons.css' ), array(), UNCANNY_REPORTING_VERSION );
			wp_enqueue_style( 'tclr-select2', Config::get_admin_css( 'select2.min.css' ), array(), UNCANNY_REPORTING_VERSION );
			wp_enqueue_style( 'tclr-backend', Config::get_admin_css( 'admin-style.css' ), array(), UNCANNY_REPORTING_VERSION );
		}

		if ( $page && 'uncanny-tincanny-site-check' === $page ) {
			// Get Tin Canny settings
			$tincanny_settings = \TINCANNYSNC\Admin\Options::get_options();

			// API data
			$reporting_api_setup = array(
				'root'            => home_url(),
				'nonce'           => \wp_create_nonce( 'tincanny-module' ),
				'isAdmin'         => is_admin(),
				'editUsers'       => current_user_can( 'edit_users' ),
				'optimized_build' => '1',
				'test_user_email' => wp_get_current_user()->user_email,
				'page'            => 'reporting',
				'showTinCanTab'   => 1 === (int) $tincanny_settings['tinCanActivation'] ? '1' : '0',
			);

			wp_localize_script( 'uncannyowl-core', 'reportingApiSetup', $reporting_api_setup );
			wp_enqueue_script( 'uncannyowl-core' );
		}
	}

	/**
	 * Submit ticket
	 */
	public static function uo_admin_help_process() {
		if ( ultc_filter_has_var( 'tclr-send-ticket', INPUT_POST ) && check_admin_referer( 'uncanny0w1', 'tclr-send-ticket' ) ) {
			$name        = esc_html( ultc_filter_input( 'fullname', INPUT_POST ) );
			$email       = esc_html( ultc_filter_input( 'email', INPUT_POST ) );
			$website     = esc_url_raw( ultc_filter_input( 'website', INPUT_POST ) );
			$license_key = esc_html( ultc_filter_input( 'license_key', INPUT_POST ) );
			$message     = esc_html( ultc_filter_input( 'message', INPUT_POST ) );
			$siteinfo    = ultc_filter_has_var( 'siteinfo', INPUT_POST ) ? stripslashes( $_POST['siteinfo'] ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			$message     = '<h3>Message:</h3><br/>' . wpautop( $message );
			if ( ! empty( $website ) ) {
				$message .= '<hr /><strong>Website:</strong> ' . $website;
			}
			if ( ! empty( $license_key ) ) {
				$message .= '<hr /><strong>License:</strong> <a href="https://www.uncannyowl.com/wp-admin/edit.php?post_type=download&page=edd-licenses&s=' . $license_key . '" target="_blank">' . $license_key . '</a>';
			}
			if ( ultc_filter_has_var( 'site-data', INPUT_POST ) && 'yes' === sanitize_text_field( ultc_filter_input( 'site-data', INPUT_POST ) ) ) {
				$message = "$message<hr /><h3>User Site Information:</h3><br />{$siteinfo}";
			}

			$to        = 'support.41077.bb1dda3d33afb598@helpscout.net';
			$subject   = esc_html( ultc_filter_input( 'subject', INPUT_POST ) );
			$headers   = array( 'Content-Type: text/html; charset=UTF-8' );
			$headers[] = 'From: ' . $name . ' <' . $email . '>';
			$headers[] = 'Reply-To:' . $name . ' <' . $email . '>';
			wp_mail( $to, $subject, $message, $headers );
			if ( ultc_filter_has_var( 'page', INPUT_POST ) ) {
				$url = admin_url( 'admin.php' ) . '?page=' . esc_html( ultc_filter_input( 'page', INPUT_POST ) ) . '&sent=true&wpnonce=' . wp_create_nonce();
				wp_safe_redirect( $url );
				exit;
			}
		}
	}

	/**
	 * Register reporting license option
	 *
	 * @return void
	 */
	public static function uo_reporting_register_option() {
		// creates our settings in the options table
		register_setting(
			'uo_reporting_license',
			'uo_reporting_license_key',
			array(
				__CLASS__,
				'uo_reporting_sanitize_license',
			)
		);
	}


	/**
	 * Sanitize the license key
	 *
	 * @param $new
	 *
	 * @return mixed
	 */
	public static function uo_reporting_sanitize_license( $new ) {
		$old = get_option( 'uo_reporting_license_key' );
		if ( $old && $old !== $new ) {
			delete_option( 'uo_reporting_license_status' ); // new license has been entered, so must reactivate
		}

		return $new;
	}


	/**
	 * Class autoloader
	 *
	 * @static
	 *
	 * @param $class
	 */
	public static function auto_loader( $class ) {

		// Remove Class's namespace eg: my_namespace/MyClassName to MyClassName
		$class = str_replace( self::get_namespace(), '', $class );
		$class = str_replace( '\\', '', $class );

		// First Character of class name to lowercase eg: MyClassName to myClassName
		$class_to_filename = lcfirst( $class );

		// Split class name on upper case letter eg: myClassName to array( 'my', 'Class', 'Name')
		$split_class_to_filename = preg_split( '#([A-Z][^A-Z]*)#', $class_to_filename, 0, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY );

		if ( 1 <= count( $split_class_to_filename ) ) {
			// Split class name to hyphenated name eg: array( 'my', 'Class', 'Name') to my-Class-Name
			$class_to_filename = implode( '-', $split_class_to_filename );
		}

		// Create file name that will be loaded from the classes directory eg: my-Class-Name to my-class-name.php
		$file_name = 'uncanny-reporting/' . strtolower( $class_to_filename ) . '.php';
		if ( file_exists( dirname( __FILE__ ) . '/' . $file_name ) ) {
			include_once $file_name;
		}

		// Create file name that will be loaded from the classes directory eg: my-Class-Name to my-class-name.php
		$file_name = 'uncanny-question-analysis-report/' . strtolower( $class_to_filename ) . '.php';
		if ( file_exists( dirname( __FILE__ ) . '/' . $file_name ) ) {
			include_once $file_name;
		}

		$file_name = 'tincanny-zip-uploader/' . strtolower( $class_to_filename ) . '.php';
		if ( file_exists( dirname( __FILE__ ) . '/' . $file_name ) ) {
			include_once $file_name;
		}

		$file_name = strtolower( $class_to_filename ) . '.php';
		if ( file_exists( dirname( __FILE__ ) . '/' . $file_name ) ) {
			include_once $file_name;
		}

	}
}





